//
//=======================================================================
// Copyright 1997, 1998, 1999, 2000 University of Notre Dame.
// Authors: Andrew Lumsdaine, Lie-Quan Lee, Jeremy G. Siek
//
// This file is part of the Generic Graph Component Library
//
// You should have received a copy of the License Agreement for the
// Generic Graph Component Library along with the software;  see the
// file LICENSE.  If not, contact Office of Research, University of Notre
// Dame, Notre Dame, IN  46556.
//
// Permission to modify the code and to distribute modified code is
// granted, provided the text of this NOTICE is retained, a notice that
// the code was modified is included with the above COPYRIGHT NOTICE and
// with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
// file is distributed with the modified code.
//
// LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
// By way of example, but not limitation, Licensor MAKES NO
// REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
// PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
// OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
// OR OTHER RIGHTS.
//=======================================================================
//
//
// Revision History:
//   01 April 2001: Modified to use new <boost/limits.hpp> header. (JMaddock)
//
#ifndef BOOST_GRAPH_DETAIL_BUCKET_SORTER_HPP
#define BOOST_GRAPH_DETAIL_BUCKET_SORTER_HPP

#include <vector>
#include <boost/limits.hpp>

namespace boost {

  template <class BucketType, class ValueType, class Bucket, class ID>
  class bucket_sorter {
  public:
    typedef BucketType bucket_type;
    typedef ValueType  value_type;
    typedef typename std::vector<value_type>::size_type size_type;
    
    bucket_sorter(size_type _length, bucket_type _max_bucket, 
                  const Bucket& _bucket = Bucket(), const ID& _id = ID()) 
      : head(_max_bucket, invalid_value(size_type())),
        next(_length, invalid_value(value_type())), 
        prev(_length, invalid_value(value_type())),
        id_to_value(_length),
        bucket(_bucket), id(_id) { }
    
    void remove(const value_type& x) {
      const size_type i = id[x];
      const size_type& next_node = next[i];
      const size_type& prev_node = prev[i];
    
      //check if i is the end of the bucket list 
      if ( next_node != invalid_value(size_type()) )
        prev[next_node] = prev_node; 
      //check if i is the begin of the bucket list
      if ( prev_node != invalid_value(size_type()) )
        next[prev_node] = next_node;
      else //need update head of current bucket list
        head[ bucket[x] ] = next_node;
    }

    void push(const value_type& x) {
      id_to_value[id[x]] = x;
      (*this)[bucket[x]].push(x);
    }
    
    void update(const value_type& x) {
      remove(x);
      (*this)[bucket[x]].push(x);
    }
    //  private: 
    //    with KCC, the nested stack class is having access problems
    //    despite the friend decl.
    static size_type invalid_value(size_type) {
      return std::numeric_limits<size_type>::max();
    }
    
    typedef typename std::vector<size_type>::iterator Iter;
    typedef typename std::vector<value_type>::iterator IterV;
    
  public:
    friend class stack;

    class stack {
    public:
      stack(bucket_type _bucket_id, Iter h, Iter n, Iter p, IterV v,
            const ID& _id = ID() )
      : bucket_id(_bucket_id), head(h), next(n), prev(p), value(v), id(_id) {}
      
      void push(const value_type& x) {
        const size_type new_head = id[x];
        const size_type current = head[bucket_id];
        if ( current != invalid_value(size_type()) )
          prev[current] = new_head;
        prev[new_head] = invalid_value(size_type());
        next[new_head] = current;
        head[bucket_id] = new_head;
      }
      
      void pop() {
        size_type current = head[bucket_id];
        size_type next_node = next[current];
        head[bucket_id] = next_node;
        if ( next_node != invalid_value(next_node) )
          prev[next_node] = invalid_value(next_node);
      }
      
      value_type& top() {
        return value[ head[bucket_id] ];
      }
      const value_type& top() const {
        return value[ head[bucket_id] ];
      }
      
      bool empty() const {
        return head[bucket_id] == invalid_value(size_type());
      }
      
    private:
      bucket_type bucket_id;
      Iter head;
      Iter next;
      Iter prev;
      IterV value;
      ID id;
    };
    
    stack operator[](const bucket_type& i) {
      return stack(i, head.begin(), next.begin(), prev.begin(),
                   id_to_value.begin(), id );
    }
    
  protected:
    std::vector<size_type>   head;
    std::vector<size_type>   next;
    std::vector<size_type>   prev;
    std::vector<value_type>  id_to_value;
    Bucket bucket;
    ID id;
  };
  
}

#endif
