//  Copyright Takuma Yoshimura 2024.
//  Copyright Matt Borland 2024.
//  Use, modification and distribution are subject to the
//  Boost Software License, Version 1.0. (See accompanying file
//  LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#define BOOST_TEST_MAIN
#define BOOST_TEST_MODULE StatsHoltsmarkTest
#include <boost/math/tools/config.hpp>
#include <boost/test/included/unit_test.hpp>
#include <boost/test/tools/floating_point_comparison.hpp>

#include <boost/math/distributions/holtsmark.hpp>

#if __has_include(<stdfloat>)
# include <stdfloat>
#endif

using boost::math::holtsmark_distribution;

#ifndef BOOST_MATH_HAS_GPU_SUPPORT
#include <boost/multiprecision/cpp_bin_float.hpp>
using boost::multiprecision::cpp_bin_float_quad;
#endif

template<class RealType, int N>
void do_test_holtsmark_pdf(){
    //
    // Basic sanity checks, tolerance is either 3 epsilon
    // expressed as a percentage:
    //
    BOOST_MATH_STD_USING
    RealType tolerance = boost::math::tools::epsilon<RealType>() * 100 * 3;

    std::cout << "Testing acurracy[%]: " << tolerance << std::endl;

    holtsmark_distribution<RealType> dist(static_cast<RealType>(0), static_cast<RealType>(1));

    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-4)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.36729417918039395222067998266923903487897550760740e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-3.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.65389736963758327689008908803579458127136270822821e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-3.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.02515191704410688567167143509210415364664018836038e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-3.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.51083986231955529936787758130352472694082331202869e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-3)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.15094236163249353135030241188004077293096105502542e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-2.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.99750209903363198419241505065146206315152726747464e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-2.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.11488945306717663129360225856869217115733169200098e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-2.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.57635410598104651856363821355027691095093972951943e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-2)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.45396231261375200568114750897618690566092315194568e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-1.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.56644599840900478087175884712634478003230341866094e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-1.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.07991505579130717014680432847812811882295188855215e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-1.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.21504440259916207727077397273468920426729181666284e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-1.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.36133628073378183373326886775069575640127303211029e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-1.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.51745548085348400860371488668500734429223868343929e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-1.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.68134487107062900924723590620591092812119992658420e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-1.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.85018067925573560771430043931430243630326746823000e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-1)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.02038159607840130388931544845552929991729709746772e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.9375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.10468477092312109723487937526691724501188944561469e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.18768434960425041116444711570747254236793293668156e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.8125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.26876533945661456653252880545328465490361724757011e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.34729279420630671501163324236373491976559833675371e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.6875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.42262031889641529582832839126322520342094549884605e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.49409942058625893002692983261905908575520500298169e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.5625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.56108951988457608160176526934999395388199046235972e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.62296840354090035789597147663858548813023471783846e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.4375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.67914286936215258107106284322230987052088197806732e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.72905929336501384188469491706325278582273134014420e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.3125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.77221383677408177290219591627664575751370295766362e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.80816200831657887972174538033323818453141437738815e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.1875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.83652730587590691770044766018047580182775092180058e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.85700868106012048340895194764008089858950037497619e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.0625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.86938660017923959009561433879636172948429661600328e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.87352751452164445024482162286994868261727837966217e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.0625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.86938660017923959009561433879636172948429661600328e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.85700868106012048340895194764008089858950037497619e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.1875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.83652730587590691770044766018047580182775092180058e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.80816200831657887972174538033323818453141437738815e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.3125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.77221383677408177290219591627664575751370295766362e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.72905929336501384188469491706325278582273134014420e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.4375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.67914286936215258107106284322230987052088197806732e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.62296840354090035789597147663858548813023471783846e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.5625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.56108951988457608160176526934999395388199046235972e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.49409942058625893002692983261905908575520500298169e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.6875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.42262031889641529582832839126322520342094549884605e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.34729279420630671501163324236373491976559833675371e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.8125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.26876533945661456653252880545328465490361724757011e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.18768434960425041116444711570747254236793293668156e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.9375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.10468477092312109723487937526691724501188944561469e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.02038159607840130388931544845552929991729709746772e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.85018067925573560771430043931430243630326746823000e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.68134487107062900924723590620591092812119992658420e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.51745548085348400860371488668500734429223868343929e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.36133628073378183373326886775069575640127303211029e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.21504440259916207727077397273468920426729181666284e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.07991505579130717014680432847812811882295188855215e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.56644599840900478087175884712634478003230341866094e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(2)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.45396231261375200568114750897618690566092315194568e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(2.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.57635410598104651856363821355027691095093972951943e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(2.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.11488945306717663129360225856869217115733169200098e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(2.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.99750209903363198419241505065146206315152726747464e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(3)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.15094236163249353135030241188004077293096105502542e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(3.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.51083986231955529936787758130352472694082331202869e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(3.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.02515191704410688567167143509210415364664018836038e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(3.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.65389736963758327689008908803579458127136270822821e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(4)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.36729417918039395222067998266923903487897550760740e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(4.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.66754576694881156512310862711445437434536539665220e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.11173604765480684115169149814306652067264621028219e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(5.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.40661030133408839114013026681116038222350468593972e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(6)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.22346322254737897976061662951210609505224644527458e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(6.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.37441612177611972649583292419049409747033180591176e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(7)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.74744460065068339118467750880557414156817094839507e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(7.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.27304026309950351561235691054710976330201642992015e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(8)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.90649774685568282389553481307707005424869182652166e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(9)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.38767874243521145729361398147898730301527980427894e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(10)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.04777602492944046116536405570429208835346612226751e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(11)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.14427312838534546510639939284636768866987456541955e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(12)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.48125490715334110982111302156190976484065179596964e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(13)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.25926848543648125740563360613612653857829343973075e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(14)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.33819608589296696976594994753563070519825392999328e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(15)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.62882342819480981578669644070664737082208227167328e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(16)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.07231582988207590928480356376941073734041404814646e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(18)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.27006701837892764913571864450685488819589225207004e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(20)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.73366906892470965030093227280098921046549874789447e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(22)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.35957320021440204694920969306620767773792096574145e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(24)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.08962051920666002556891360954800856160496745325982e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(26)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.89257005093845424113694708405490444536741708307432e-5), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(28)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.36985573727704971748849427466806366268717813920884e-5), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(30)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.18908057108330090847508167263164576897080536147998e-5), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(32)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.25741312407933720816582583160953651639222885045617e-5), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(36)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.90521803068812235229266578780465914597024834788499e-5), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(40)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.99440098605222381975051163529379005706849379136631e-5), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(44)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.35556983515831713649398537010595127693244595946631e-5), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(48)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.89252591391344813399503397931059007207167187335384e-5), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(52)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.54761819718387004182264813450582914982582239574760e-5), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(56)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.28473144505453904388319480756963924103363111590946e-5), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(60)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.08038477619830474574170723303117982155643976342020e-5), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(64)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.18821097500148811906668422616947735567544539343910e-6), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(72)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.83770756909753597744679176403606964363170824955403e-6), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(80)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.25030682765210472090258978149127752580598842866547e-6), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(88)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.13470581474698689445958562292807701026370826971490e-6), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(96)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.32481553840568432886657947564022812389544240910245e-6), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(104)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.72079249234488105623103527050493983216840056389979e-6), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(112)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.25993583492605042781181768463472827114252734385354e-6), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(120)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.90139785097711606987697985776461422964227374409867e-6), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(128)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.61772178749879746562419160426660201168856327635754e-6), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(256)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.85568203412051586615998003517909621954036193765824e-7), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(512)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.04563780918059698346140819518179336362098891429611e-8), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1024)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.91792339891869109849236922708731694322153177924824e-9), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 11)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.57638176923363231335450711092364627257129748508825e-9), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 12)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.78661356398095539721893167504764083609118624639856e-10), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 13)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.92604459926540711296353370623384343787650705186643e-11), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 14)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.70807462044367666228538283653300633948804226008283e-12), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 15)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.53938313936621297172285881159282265471330118114236e-12), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 16)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.72126969520581558576219170420297785383800313247716e-13), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 17)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.81057004571925961169117504945825287978291651680376e-14), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 18)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.50396638201843310535816082381660912483279969324341e-15), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 19)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.50330305081854272655464104542050109297998338121296e-15), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 20)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.65748943908780839905864199570025333227786061957216e-16), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 21)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.69782199924985143903444545973473854205301886566784e-17), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 22)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.30465447555059221873312238790689561215989143051468e-18), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 23)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.46806937341565301230797169380667244412810953871153e-18), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 24)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.59520452276585659227559942516968833301265805493281e-19), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.58771679139660099100514647692148711295768774585921e-20), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 26)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.11001413331373018059574227843565648889473010676660e-21), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 27)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.43366149729083958029796238740037446793377590904146e-21), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 28)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.53437941664766678806680232945039585344994407777604e-22), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 29)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.48019217902582750507529988918628139079404963189635e-23), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 30)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.91993567701892984811136675864172386794777828712695e-24), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 31)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.40006005594525679129869441468227938628239752074460e-24), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 32)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.47497989906821913751146663525981445654490373951224e-25), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 34)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.73431218458810807033921781897016124929992555367139e-27), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 36)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.41697255768378077454412606000376285104775580444420e-28), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 38)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.55303924276181374957661287953452523234530316565188e-30), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 40)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.36032476336306675100543445227730858361169873925804e-31), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 42)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.37601488550958357902586661939110889053699438042016e-33), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 44)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.30500465172174486774768816059192932509423715417166e-34), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 46)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.20313953663045271143891020576861031160487434865093e-36), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 48)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.25098110519701647231401040429952006498171313622520e-37), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 50)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.03431595374067647597712273413788666502302011992057e-39), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 52)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.19822373554396139874268836303925702273981755123190e-40), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 54)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.86944917357487937107083766130282172728072819621788e-42), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 56)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.14670286674214980345963428973545770896346187182759e-43), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 58)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.70844645856921813581135618689921390269372237700480e-45), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 60)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.09638951830288066744104877057303202280205793138993e-46), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 62)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.55121724469650208575327739326222025805731763269349e-48), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 64)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.04725538896765690179789918481715848637732139291593e-49), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 68)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.99927284078872744316201092259372825667966088624440e-52), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 72)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.95241488358274164371290129159422806319923560612665e-55), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 76)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.90665515974877113643838016757246988460215123083510e-58), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 80)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.86196792944215931292810563239498984002947733280447e-61), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 84)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.81832805609585870403135315663573226135611325562940e-64), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 88)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.77571099228111201565561831702708228641500206234884e-67), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 92)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.73409276589952345278868976272176004532614982149466e-70), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 96)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.69344996669875337186395484640796879426380291665101e-73), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 100)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.65375973310425133971089340469528202564824480281467e-76), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 104)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.61499973935962044893641934052273635317211406169375e-79), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 108)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.57714818296837934466447201222923471989464263831856e-82), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 112)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.54018377243005795377389844944261203114711195148214e-85), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 116)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.50408571526372847048232270453380081166710151511926e-88), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 120)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.46883370631223483445539326614628985514365382335866e-91), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 124)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.43440791632054183052284498647098618666372443687369e-94), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 128)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.40078898078177913136996580710057244791379339538446e-97), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 128)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.40078898078177913136996580710057244791379339538446e-97), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 136)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.33589647367647088181492405614907498160724010027357e-103), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 144)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.27401015632292831593983083357722757492755899455411e-109), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 152)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.21499076492588836282713969571802861683612727599536e-115), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 160)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.15870548718060337336267442294886457141506888961350e-121), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 168)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.10502766340313279472606127066503960744387520753240e-127), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 176)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.05383650150597838852506758753303490394961853745690e-133), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 184)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.00501680517766798832422980073264589686357358689966e-139), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 192)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.58458714654605854343633461697240731109212481403024e-146), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 200)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.14057459501844267219193898865929347142422181513809e-152), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 208)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.71713122846454875201410197130135867254659825815019e-158), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 216)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.31330416532950282288942525034080378775272203269023e-164), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 224)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.92818466694784433640425229105072382712623790043853e-170), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 232)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.56090609259399827614236096482345946037887373012402e-176), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 240)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.21064194926643207182155701143594690359008191120531e-182), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 248)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.87660403181689459974437428611368837698944274063617e-188), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 256)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.55804064923943958258092335330361211489624284804933e-194), tolerance);
}

template<class RealType, int N>
void do_test_holtsmark_cdf() {
    //
    // Basic sanity checks, tolerance is either 3 epsilon
    // expressed as a percentage:
    //
    BOOST_MATH_STD_USING
    RealType tolerance = boost::math::tools::epsilon<RealType>() * 100 * 3;

    std::cout << "Testing acurracy[%]: " << tolerance << std::endl;

    holtsmark_distribution<RealType> dist(static_cast<RealType>(0), static_cast<RealType>(1));

    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-4)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.05754562114095147060025732340404110273918791128410e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-3.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.43368449353064312968788810247763526151579823896375e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-3.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.89152652981222928500336992235218785823347685619486e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-3.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.45576815036790322986175114501424442292398775002753e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-3)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.15978035591850473786135203626730317231876547935253e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-2.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.04840867577215219812315602202859291564396444891271e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-2.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.18107352643844092577238041212187564083206726822056e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-2.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.63452659158632535349648745626466375370661702427253e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-2)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.05039829654829170780787685299557006719608839147146e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-1.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.16290125976286647650587923507379034571826555851855e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-1.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.29006094814340314434551699445775276932192677014796e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-1.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.43337488600334723245402521343673257218591019586678e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-1.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.59428804524998597404889957316726146572475641692056e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-1.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.77411943695232738645173996334900914289694617802622e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-1.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.97397674713122675501528813572074060920189377220435e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-1.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.19466281873758062089629668872911218009781881016957e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-1)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.43657975600729535499895880792984203156689462603233e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.9375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.56549331000915958092357114949075709547374035120716e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.69963821328456787766164921360924665804748003656962e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.8125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.83891389704828738280543932012082190429368746897774e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.98318069008956501931929973721256792184639224594009e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.6875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.13225883262651243874304479279543149837428487068880e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.28592804901375835801811223649103787689862120396348e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.5625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.44392772657854742558827130471487286937515502098876e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.60595773518728397925852903878144782668636743128847e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.4375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.77167990746831553875195022302050315427406387091454e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.94072018335986771952143399033763464047462959349960e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.3125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.11267140533666880513239944176560287064005137544504e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.28709673289702231328042170648699212093305785288837e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.1875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.46353362731996489993314637378816744341919505345430e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.64149834100915287958166424201293669944118353975980e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.0625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.82049083059559169016945833205795772857456974521089e-1), tolerance);
    BOOST_CHECK_EQUAL(cdf(dist, static_cast<RealType>(0)), static_cast<RealType>(0.5));
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.0625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.17950916940440830983054166794204227142543025478911e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.35850165899084712041833575798706330055881646024020e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.1875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.53646637268003510006685362621183255658080494654570e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.71290326710297768671957829351300787906694214711163e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.3125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.88732859466333119486760055823439712935994862455496e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.05927981664013228047856600966236535952537040650040e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.4375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.22832009253168446124804977697949684572593612908546e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.39404226481271602074147096121855217331363256871153e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.5625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.55607227342145257441172869528512713062484497901124e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.71407195098624164198188776350896212310137879603652e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.6875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.86774116737348756125695520720456850162571512931120e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.01681930991043498068070026278743207815360775405991e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.8125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.16108610295171261719456067987917809570631253102226e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.30036178671543212233835078639075334195251996343038e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.9375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.43450668999084041907642885050924290452625964879284e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(1)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.56342024399270464500104119207015796843310537396767e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(1.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.80533718126241937910370331127088781990218118983044e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(1.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.02602325286877324498471186427925939079810622779565e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(1.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.22588056304767261354826003665099085710305382197378e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(1.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.40571195475001402595110042683273853427524358307944e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(1.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.56662511399665276754597478656326742781408980413322e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(1.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.70993905185659685565448300554224723067807322985204e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(1.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.83709874023713352349412076492620965428173444148145e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(2)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.94960170345170829219212314700442993280391160852854e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(2.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.13654734084136746465035125437353362462933829757275e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(2.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.28189264735615590742276195878781243591679327317794e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(2.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.39515913242278478018768439779714070843560355510873e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(3)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.48402196440814952621386479637326968276812345206475e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(3.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.55442318496320967701382488549857555770760122499725e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(3.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.61084734701877707149966300776478121417665231438051e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(3.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.65663155064693568703121118975223647384842017610363e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(4)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.69424543788590485293997426765959588972608120887159e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(4.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.75181239051510561995724918481437181225561682663730e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.79330912859883809209439632732487322514831404016950e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(5.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.82433333964997258238374422700352745232108521574338e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(6)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.84823776850456273960749947374965581355459282720918e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(6.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.86712078422837181984231917376692653955733800370589e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(7)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.88234978933381804514525314568892574021644736001062e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(7.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.89484817977790488434819448808352507951600536883956e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(8)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.90525915297517643347206332876433307912297417265919e-1), tolerance);
}

template<class RealType, int N>
void do_test_holtsmark_ccdf() {
    //
    // Basic sanity checks, tolerance is either 3 epsilon
    // expressed as a percentage:
    //
    BOOST_MATH_STD_USING
    RealType tolerance = boost::math::tools::epsilon<RealType>() * 100 * 3;

    std::cout << "Testing acurracy[%]: " << tolerance << std::endl;

    holtsmark_distribution<RealType> dist(static_cast<RealType>(0), static_cast<RealType>(1));

    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-2))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.94960170345170829219212314700442993280391160852854e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-1.875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.83709874023713352349412076492620965428173444148145e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-1.75))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.70993905185659685565448300554224723067807322985204e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-1.625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.56662511399665276754597478656326742781408980413322e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-1.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.40571195475001402595110042683273853427524358307944e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-1.375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.22588056304767261354826003665099085710305382197378e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-1.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.02602325286877324498471186427925939079810622779565e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-1.125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.80533718126241937910370331127088781990218118983044e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-1))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.56342024399270464500104119207015796843310537396767e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.9375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.43450668999084041907642885050924290452625964879284e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.30036178671543212233835078639075334195251996343038e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.8125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.16108610295171261719456067987917809570631253102226e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.75))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.01681930991043498068070026278743207815360775405991e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.6875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.86774116737348756125695520720456850162571512931120e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.71407195098624164198188776350896212310137879603652e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.5625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.55607227342145257441172869528512713062484497901124e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.39404226481271602074147096121855217331363256871153e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.4375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.22832009253168446124804977697949684572593612908546e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.05927981664013228047856600966236535952537040650040e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.3125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.88732859466333119486760055823439712935994862455496e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.71290326710297768671957829351300787906694214711163e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.1875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.53646637268003510006685362621183255658080494654570e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.35850165899084712041833575798706330055881646024020e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.0625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.17950916940440830983054166794204227142543025478911e-1), tolerance);
    BOOST_CHECK_EQUAL(cdf(complement(dist, static_cast<RealType>(0))), static_cast<RealType>(0.5));
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.0625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.82049083059559169016945833205795772857456974521089e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.64149834100915287958166424201293669944118353975980e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.1875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.46353362731996489993314637378816744341919505345430e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.28709673289702231328042170648699212093305785288837e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.3125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.11267140533666880513239944176560287064005137544504e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.94072018335986771952143399033763464047462959349960e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.4375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.77167990746831553875195022302050315427406387091454e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.60595773518728397925852903878144782668636743128847e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.5625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.44392772657854742558827130471487286937515502098876e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.28592804901375835801811223649103787689862120396348e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.6875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.13225883262651243874304479279543149837428487068880e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.75))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.98318069008956501931929973721256792184639224594009e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.8125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.83891389704828738280543932012082190429368746897774e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.69963821328456787766164921360924665804748003656962e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.9375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.56549331000915958092357114949075709547374035120716e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.43657975600729535499895880792984203156689462603233e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1.125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.19466281873758062089629668872911218009781881016957e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.97397674713122675501528813572074060920189377220435e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1.375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.77411943695232738645173996334900914289694617802622e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.59428804524998597404889957316726146572475641692056e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1.625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.43337488600334723245402521343673257218591019586678e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1.75))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.29006094814340314434551699445775276932192677014796e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1.875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.16290125976286647650587923507379034571826555851855e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(2))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.05039829654829170780787685299557006719608839147146e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(2.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.63452659158632535349648745626466375370661702427253e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(2.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.18107352643844092577238041212187564083206726822056e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(2.75))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.04840867577215219812315602202859291564396444891271e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(3))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.15978035591850473786135203626730317231876547935253e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(3.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.45576815036790322986175114501424442292398775002753e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(3.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.89152652981222928500336992235218785823347685619486e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(3.75))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.43368449353064312968788810247763526151579823896375e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(4))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.05754562114095147060025732340404110273918791128410e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(4.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.48187609484894380042750815185628187744383173362701e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.06690871401161907905603672675126774851685959830498e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(5.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.75666660350027417616255772996472547678914784256620e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(6))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.51762231495437260392500526250344186445407172790817e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(6.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.32879215771628180157680826233073460442661996294112e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(7))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.17650210666181954854746854311074259783552639989383e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(7.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.05151820222095115651805511916474920483994631160437e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(8))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.47408470248235665279366712356669208770258273408139e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(9))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.84633233633455623701916449563937321750458546930453e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(10))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.63980919776847052871529866313101745651230650390958e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(11))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.71560271091813267790502831701953317735310509411390e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(12))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.98877952779840995427042841099634486251917561421235e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(13))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.40474538743684446341559852404247422921805516802754e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(14))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.92694975034713587589009244180464471384939103569360e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(15))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.53008216069181937774819568728358865566934398876406e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(16))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.19610991747326725339429696634365931201323237447865e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(18))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.66754697245984312281273324441306444625571550964874e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(20))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.27055303995134836245432119316955904009319977641407e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(22))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.96335385591609284864855802297989450933958012331026e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(24))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.71983068404058084817697571696065106175519154567134e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(26))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.52289416073896095009461377823763903270627907120760e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(28))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.36093802515022944719982175339249822123873546752029e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(30))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.22583027731897389283291308558313649709444449306352e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(32))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.11172037056341396583040940446061500559620114648017e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(36))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.30343276748924876383449972567328892540120922780153e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(40))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.93479451609843506274951848311637162210971363421001e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(44))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.87195324030611990483144403072151789810819131065125e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(48))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.02706331251660616582062908165140543120249377843861e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(52))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.34227281260431186934023112980425214646324423530835e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(56))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.77808682390572890160969127059116197868923042197157e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(60))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.30671561098656496307105837586195479010992455366704e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(64))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.90809577465797739887717333706777298095814323854594e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(72))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.27353325758395877256315285908308787899438871193344e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(80))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.79392283157821028307612118603031347679870090601645e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(88))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.42100764912263851750335808792156725434230560510519e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(96))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.12427366298380215713674931256380175628765437052256e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(104))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.88357904993849478063900853297202135211787374336745e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(112))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.68514784935653217641685603828708555083243580835252e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(120))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.51927236784831507287987712585213807190192778503378e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(128))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.37893523202524132194116984313653211252949573780743e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(256))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.87179879353850251536193247663317269917137569329013e-5), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(512))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.72200715673082274403568012456413425331659480927047e-5), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1024))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.08767256634253255597053159005214917567906016620942e-6), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 11))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.15224951834965340756402503731619739933629215689572e-6), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 12))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.60926645137475638433041259135060814007792759468100e-7), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 13))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.69027136710097153757239920401079588116557633592788e-7), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 14))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.51153240121827639330112195094153363441740650435701e-8), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 15))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.36283287602527689849064009106938913094716520594050e-8), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 16))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.18894075854698171408179022464997418218329601220535e-8), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 17))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.20354010552543796231728241342645692784842686349965e-9), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 18))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.48617582449547816069323420820736862370145659607645e-9), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 19))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.25442497729649541267939242007563203416972765165476e-10), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 20))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.85771976129306035185873711236944017216967773980517e-10), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 21))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.56803119746276025902637885330726865801883592203156e-11), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 22))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.32214969859659307190062902673645385449168267230206e-11), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 23))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.21003899305378486738410997502574389330326609057126e-12), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 24))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.90268712277390815798081242324390081166198190434103e-12), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.02625487407274396069298698869802596448489279633645e-12), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 26))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.62835890339366126281650716118851052269284048205938e-13), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 27))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.28281859258171445903392888079435183340041265714998e-13), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 28))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.53544862923055721373031196622890601888700626111182e-14), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 29))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.60352324072570315319362346398633880353985161880622e-14), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 30))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.66931078653639661641440656550426788294070266782671e-15), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 31))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.00440405090690395389846895694672344247801107283103e-15), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 32))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.08663848317021453602605776312368607691806298813143e-16), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 34))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.85829810396272422714320494875293186097734524880995e-17), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 36))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.10728726299533984178525425523566315206520781873938e-17), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 38))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.38410907874417469494912307476988489629927133322828e-18), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 40))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.73013634843021835192352185216943643167993641440545e-19), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 42))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.16267043553777293728520200407227686276581114275508e-20), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 44))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.70333804442221617119725245647479628568092558840977e-21), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 46))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.37917255552777021393262025049731570198920941773050e-22), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 48))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.22396569440971276740578385685661655637545268535833e-23), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 50))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.27995711801214095925566865602936005935821644314758e-24), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 52))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.59994639751517619906934188799897966230541134026964e-25), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 54))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.24993299689397024883663924561783076352358304956329e-26), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 56))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.03124162461174628110457931016502737959110130110786e-26), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 58))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.28905203076468285138072404465359649544991622324619e-27), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 60))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.61131503845585356422590504127751316165005771606733e-28), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 62))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.01414379806981695528238129932509731805283190086691e-29), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 64))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.51767974758727119410297662380140381412701796292469e-30), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 68))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.93387460560511124078590097461169759773383672872924e-32), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 72))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.14667907125798631372797027282887330061342582520339e-34), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 76))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.60418604884060361519995355129506804305208883665969e-36), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 80))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.50065407013134431487499274238985426822797191535889e-37), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 84))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.34477198458022549199217615998414729133523646527342e-39), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 88))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.66370622590660233123777524997523014264365635070860e-41), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 92))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.72454097797906614255902382808629709787906142136400e-43), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 96))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.94459527809229084774847473138483921543599314796577e-45), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 100))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.39759301220192044496069917677888112741187383092503e-46), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 104))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.18373908156550069525109246371700176158105285841693e-48), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 108))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.41209231494609483632983197455781525247039509121778e-50), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 112))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.33139424210327318176536246024658633198499233002634e-52), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 116))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.33030350328636434650837884413529114372655051566613e-54), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 120))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.30160992238849442914193419439613924120727351807283e-55), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 124))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.03376550373202254553427217874396756438636487198880e-57), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 128))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.17775859958128522739730027928744931935369511248250e-59), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 128))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.17775859958128522739730027928744931935369511248250e-59), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 136))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.75819970600899713720044013497912431482835720820923e-63), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 144))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.89409172509985281669932620482888777217489189653546e-66), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 152))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.62424737573206254076983936725802678753635716927602e-70), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 160))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.12896664446583558124263656427197919617586844953028e-73), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 168))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.75626622184041889951815567449213670941374133186103e-77), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 176))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.72916558066508520390174725217806813821714192348884e-81), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 184))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.64286269059206181735882501273878616655691941491427e-84), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 192))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.01089524070327592128619387875680216444560404031804e-88), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 200))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.79222470874823222970262177430859903429102548905771e-92), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 208))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.39067986053423638420474164411831031110620739478948e-95), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 216))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.83662075325741304737485752958571853297413914743525e-99), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 224))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.42495623858823560726925232655901331371438944029181e-102), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 232))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.47889706686580958805969806288821609793552109446243e-106), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 240))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.49340104215285543959887222384818383285039329702741e-110), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 248))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.07358423880684947255831841402543550606699055103208e-113), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 256))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.06246152052453484511308206549178590348386364998067e-117), tolerance);

    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 600))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.35984768491636685338723260976696756010081951066709e-272), tolerance);

    // The test is terminated because x is inf after this in fp64.
    if (N <= 53) {
        return;
    }

    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 10000))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.07856314267954244808378917796336533616174400834444e-4517), tolerance);
}

template<class RealType, int N>
void do_test_holtsmark_quantile_nearzero() {
    //
    // Basic sanity checks, tolerance is either 3 epsilon
    // expressed as a percentage:
    //
    BOOST_MATH_STD_USING
    RealType tolerance = boost::math::tools::epsilon<RealType>() * 100 * 4;

    std::cout << "Testing acurracy[%]: " << tolerance << std::endl;

    holtsmark_distribution<RealType> dist(static_cast<RealType>(0), static_cast<RealType>(1));

    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.03125)), BOOST_MATH_BIG_CONSTANT(RealType, N, -3.95154561868642736172727541497862573947423988108440e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.0625)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.70076903638473389271107254016757513871156973570753e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.09375)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.14334937403136202998120100318789158621074793095204e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.78777320599109327486003359289950832456660816353247e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.15625)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.52359744792606448551215814625237535910408488746319e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.1875)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.31030403921397562363480619469818801698833367088973e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.21875)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.12887694529692805365864437467349833307816315605127e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, -9.68933181713583005208786313049614919507807226788256e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.28125)), BOOST_MATH_BIG_CONSTANT(RealType, N, -8.24180933056718074475811972779037648260058221511975e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.3125)), BOOST_MATH_BIG_CONSTANT(RealType, N, -6.90498457639035315478919560434827057700411821656397e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.34375)), BOOST_MATH_BIG_CONSTANT(RealType, N, -5.65011034985136468564856594364133703131867596374603e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, -4.45602574935161564367593993713685007358536186330925e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.40625)), BOOST_MATH_BIG_CONSTANT(RealType, N, -3.30636223130194458226267853223267146033119688836326e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.4375)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.18783126811804488668597304309230757418655445388316e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.46875)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.08910089454817618171205461377219663239656206354005e-1), tolerance);
    BOOST_CHECK_EQUAL(quantile(dist, static_cast<RealType>(0.5)), static_cast<RealType>(0));
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.50390625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.35942281074760935023421465350943213907657016740203e-2), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.5078125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.71903117534558589826852519194663375075753566669865e-2), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.51171875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.07901080568320221484687881885423066805647774437405e-2), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.515625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.43954773006081419776089446583406196027634755818438e-2), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.51953125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.80082845222988104915780076816665811376325257794804e-2), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.5234375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.16304011143799846843111212111260495740955724460844e-2), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.52734375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.52637064382003902226333974540096053392566128735022e-2), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.53125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.08910089454817618171205461377219663239656206354005e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.53515625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.22571450376289000001278843371459783620761964026442e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.5390625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.36249702341028008589045826702986410397012912904709e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.54296875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.49946773116932301309732751744953722231029126346486e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.546875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.63664606836100199300226165836560082654887238338208e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.55078125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.77405165765079089783419383641965337744707947259452e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.5546875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.91170432114732759551367174850419509782585472590891e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.55859375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.04962409893975946705863512977926741730110899605668e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.5625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.18783126811804488668597304309230757418655445388316e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.56640625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.32634636232249536218129854873203382200612639970098e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.5703125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.46519019187105731139818680492599864661432850376167e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.57421875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.60438386451527503677057942069112096470660708202704e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.578125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.74394880687856404334281468719091417527159191461011e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.58203125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.88390678663337502806483517706706303095739191330472e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.5859375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.02427993547706435128366410334899635275725704584021e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.58984375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.16509077296982960387421259319514641259915543447926e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.59375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.30636223130194458226267853223267146033119688836326e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.59765625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.44811768106176498805637646337817923388445968601566e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.6015625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.59038095808060603000739360761949422570819209911788e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.60546875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.73317639143565086527888233910690716885943557654200e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.609375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.87652883269757340156275635385373789172745768932299e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.61328125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.02046368651559365036792703758158149784555601527522e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.6171875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.16500694263927667044145946879116606537710528519644e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.62109375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.31018520948359067342384024222999190532594772267877e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.45602574935161564367593993713685007358536186330925e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.62890625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.60255651543790720669175415204007861250418903243190e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.6328125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.74980619074494545171846930824082767773201407564280e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.63671875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.89780422905541754178666731200120149076546958331875e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.640625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.04658089811438846102378009478135229217539543328542e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.64453125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.19616732518780928821098759267869073306168616762584e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.6484375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.34659554517741239172052324381351337830662490680744e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.65234375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.49789855148697728320168560667825948269915149112198e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.65625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.65011034985136468564856594364133703131867596374603e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.66015625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.80326601535777269419392897717286438879602729753825e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.6640625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.95740175290855108336930680485963056005472680796594e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.66796875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.11255496139682430686772264889247897534154814090188e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.671875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.26876430189035334589546274389116075758343949758676e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.67578125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.42606977014577414303134971466049624688623181490963e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.6796875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.58451277380488310882288163359470075488686595989134e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.68359375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.74413621465733462219129425896964561569492923310844e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.6875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.90498457639035315478919560434827057700411821656397e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.69140625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.06710401828627699619029762829527775785025847826609e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.6953125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.23054247537343446268743874344186034038860806575417e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.69921875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.39534976558556849766533672421653104101284193327489e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.703125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.56157770454041251979362066603791706893576777672076e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.70703125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.72928022860981170501224651695596294779402544635284e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.7109375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.89851352702281826568041708053406482267060399077072e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.71484375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.06933618382042868991732981731903327683778049211437e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.71875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.24180933056718074475811972779037648260058221511975e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.72265625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.41599681082196095188647911444502703373377770261645e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.7265625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.59196535747955714353182241432999465132648793939716e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.73046875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.76978478421742214185140384463517803619244579498785e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.734375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.94952819242075830879661206434669971092120406861116e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.73828125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.13127219511566873406119301005062170096048576202346e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.7421875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.31509715961740080067603245114697688184732500932588e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.74609375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.50108747080172235764142816696014524944397797022118e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.68933181713583005208786313049614919507807226788256e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.75390625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.87992350186512083533727177758662092012850600324604e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.7578125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.00729607820486228792863859054788768644744365249151e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.76171875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.02685472384747190194216544893563951741322238866541e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.765625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.04667921798768908218602656800798525125137666299761e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.76953125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.06678110853147091220341113291781102477507787382916e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.7734375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.08717260890978562986849760571609969494354300822263e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.77734375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.10786665132220733188524032283620543957825067010411e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.78125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.12887694529692805365864437467349833307816315605127e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.78515625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.15021804221161134460394215947401773984635551400974e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.7890625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.17190540651154074066537741066806130017312231661434e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.79296875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.19395549446874295273954813640902172894636549828066e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.796875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.21638584145104788446634880619086527181141939030279e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.80078125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.23921515881685008719278461299232510042094064507963e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.8046875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.26246344172387198181843263481470665031951188807903e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.80859375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.28615208934364044613153187689263023500565577152964e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.8125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.31030403921397562363480619469818801698833367088973e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.81640625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.33494391774730388327762461486718746182135887404098e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.8203125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.36009820925261741664314298719135880075160309673291e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.82421875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.38579544623528308147561770378298998291069435447743e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.828125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.41206642422644329402039008236727065491131379281346e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.83203125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.43894444498096320102061428365633543440558564852878e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.8359375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.46646559259303986239248263261753112540493595538180e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.83984375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.49466904794112146248736006028509653369456653722975e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.84375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.52359744792606448551215814625237535910408488746319e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.84765625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.55329729725622722446696215102952397443200898053734e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.8515625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.58381944212178952767489242020964875845531372433107e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.85546875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.61521961706725528436767664289542667668084392814972e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.859375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.64755907881888895242774154629775987069182281943649e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.86328125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.68090534388763421593277486421165006597881155937148e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.8671875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.71533305062559095290467926282068904086311880655794e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.87109375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.75092497130127405520149304919663885139873951807268e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.78777320599109327486003359289950832456660816353247e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.87890625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.82598059808698758538443295349421685195378444570311e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.8828125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.86566242157090907181680189290396765394233286379846e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.88671875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.90694840369852884963353039612138690706293500611801e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.890625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.94998516446439884497123835171451955479771823102329e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.89453125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.99493917772297054805704471816059443529057908424094e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.8984375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.04200039028200656353168185848438653746136511354273e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.90234375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.09138667776446768039690693058064390661207192288310e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.90625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.14334937403136202998120100318789158621074793095204e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.91015625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.19818019103302928413100774795456740418726081571239e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.9140625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.25621995785515543565780905111478616154698242356459e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.91796875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.31786976613802108820384089261971867032207069978249e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.921875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.38360533647314959687029290315016585988822268727435e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.92578125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.45399575188877505107943151191453137828469465461499e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.9296875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.52972819582716132188558325053476431645151134471514e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.93359375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.61164107343000684767083444817632028828927968917161e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.9375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.70076903638473389271107254016757513871156973570753e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.94140625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.79840522818853957979403024307775647164832128717134e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.9453125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.90618896524175502682103322803168945370341377670787e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.94921875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.02623187745561328365781914812219941666139382669240e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.953125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.16130376436882129187002225210296787778679380216571e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.95703125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.31511403805596290718635028471440503099806209998330e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.9609375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.49275168669289561003117954123304779848451307206373e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.96484375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.70139933971958385705119882168933347231988022896107e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.96875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.95154561868642736172727541497862573947423988108440e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.97265625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.25916037511354144395537844168672288266598351274466e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.9765625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.64987772024289138285743520182808805983174934278635e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.98046875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.16779337095862086311250586064141028622224308218889e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.984375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.89632696959562976212621758577393939287586784169403e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.98828125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.01689760146742972661016754806402368873114241071209e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.9921875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.02446937207505244639190533898754565373763855739835e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.99609375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.40479269149807902190125365288201718857308359786880e1), tolerance);
}

template<class RealType, int N>
void do_test_holtsmark_quantile_lower() {
    //
    // Basic sanity checks, tolerance is either 3 epsilon
    // expressed as a percentage:
    //
    BOOST_MATH_STD_USING
    RealType tolerance = boost::math::tools::epsilon<RealType>() * 100 * 3;

    std::cout << "Testing acurracy[%]: " << tolerance << std::endl;

    holtsmark_distribution<RealType> dist(static_cast<RealType>(0), static_cast<RealType>(1));

    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -3)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.78777320599109327486003359289950832456660816353247e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -4)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.70076903638473389271107254016757513871156973570753e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -5)), BOOST_MATH_BIG_CONSTANT(RealType, N, -3.95154561868642736172727541497862573947423988108440e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -6)), BOOST_MATH_BIG_CONSTANT(RealType, N, -5.89632696959562976212621758577393939287586784169403e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -7)), BOOST_MATH_BIG_CONSTANT(RealType, N, -9.02446937207505244639190533898754565373763855739835e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -8)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.40479269149807902190125365288201718857308359786880e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -10)), BOOST_MATH_BIG_CONSTANT(RealType, N, -3.48634712845212854436829850825868101882854019691874e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -12)), BOOST_MATH_BIG_CONSTANT(RealType, N, -8.75100888665534247471443047755794084714498529274899e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -14)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.20296704598194178128278789813687898595549252840676e2), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -16)), BOOST_MATH_BIG_CONSTANT(RealType, N, -5.54977454632239578277064855837334284482347724412824e2), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -20)), BOOST_MATH_BIG_CONSTANT(RealType, N, -3.52361836255645260708087122768062244544827526253646e3), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -24)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.23734752970824778029419146983721335265273533376700e4), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -28)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.42062670553125367339285994800008811139726501576815e5), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -32)), BOOST_MATH_BIG_CONSTANT(RealType, N, -9.02041713923935805934169232925253585383521252378005e5), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -40)), BOOST_MATH_BIG_CONSTANT(RealType, N, -3.63680429392084153704899843149763443252941653727485e7), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -48)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.46626761095577290791010108814328116732395911944180e9), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -56)), BOOST_MATH_BIG_CONSTANT(RealType, N, -5.91162056903107188819131273099660458893079363542047e10), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -64)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.38341606205240162091324729036756554251581053237655e12), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -80)), BOOST_MATH_BIG_CONSTANT(RealType, N, -3.87423965612020298589373786039770640757408383939246e15), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -96)), BOOST_MATH_BIG_CONSTANT(RealType, N, -6.29757143623897661671305188560924841842505858740001e18), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -112)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.02366940392761715971201478542352832996294475084257e22), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -128)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.66397326199025580692950800332626492430331068544172e25), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -160)), BOOST_MATH_BIG_CONSTANT(RealType, N, -4.39662661178492965217151228166250717006976010266469e31), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -192)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.16169688570203891305395889425358741905195655516409e38), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -224)), BOOST_MATH_BIG_CONSTANT(RealType, N, -3.06948889094299024072567114732250581789838857797378e44), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -256)), BOOST_MATH_BIG_CONSTANT(RealType, N, -8.11034458952573565292573972403491751744218292992004e50), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -256)), BOOST_MATH_BIG_CONSTANT(RealType, N, -8.11034458952573565292573972403491751744218292992004e50), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -320)), BOOST_MATH_BIG_CONSTANT(RealType, N, -5.66220760083199094068657639449976729910923748932480e63), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -384)), BOOST_MATH_BIG_CONSTANT(RealType, N, -3.95304965911372717286692649124348058110571556299024e76), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -448)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.75980725346824443030445016705237404631255977139835e89), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -512)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.92674940440883821350824836233864795634501852860731e102), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -640)), BOOST_MATH_BIG_CONSTANT(RealType, N, -9.39113744455748572074441450789762236967081046313794e127), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -768)), BOOST_MATH_BIG_CONSTANT(RealType, N, -4.57731878887639063343639480262468437174058078877170e153), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -896)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.23102338973254024994084948896750440849135610207358e179), tolerance);
}

template<class RealType, int N>
void do_test_holtsmark_quantile_upper() {
    //
    // Basic sanity checks, tolerance is either 3 epsilon
    // expressed as a percentage:
    //
    BOOST_MATH_STD_USING
    RealType tolerance = boost::math::tools::epsilon<RealType>() * 100 * 3;

    std::cout << "Testing acurracy[%]: " << tolerance << std::endl;

    holtsmark_distribution<RealType> dist(static_cast<RealType>(0), static_cast<RealType>(1));

    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -3))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.78777320599109327486003359289950832456660816353247e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -4))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.70076903638473389271107254016757513871156973570753e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.95154561868642736172727541497862573947423988108440e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -6))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.89632696959562976212621758577393939287586784169403e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -7))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.02446937207505244639190533898754565373763855739835e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -8))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.40479269149807902190125365288201718857308359786880e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -10))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.48634712845212854436829850825868101882854019691874e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -12))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.75100888665534247471443047755794084714498529274899e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -14))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.20296704598194178128278789813687898595549252840676e2), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -16))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.54977454632239578277064855837334284482347724412824e2), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -20))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.52361836255645260708087122768062244544827526253646e3), tolerance);
}

template<class RealType, int N>
void do_test_holtsmark_locscale_param() {
    //
    // Basic sanity checks, tolerance is either 3 epsilon
    // expressed as a percentage:
    //

    BOOST_MATH_STD_USING
    RealType tolerance = boost::math::tools::epsilon<RealType>() * 100 * 3;

    std::cout << "Testing acurracy[%]: " << tolerance << std::endl;

    holtsmark_distribution<RealType> dist_0_1(static_cast<RealType>(0), static_cast<RealType>(1));
    holtsmark_distribution<RealType> dist_1_3(static_cast<RealType>(1), static_cast<RealType>(3));

    BOOST_CHECK_CLOSE(entropy(dist_0_1), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.0694485051346244003155800384542166381), tolerance);
    BOOST_CHECK_CLOSE(entropy(dist_1_3), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.0694485051346244003155800384542166381) + log(static_cast<RealType>(3)), tolerance);

    BOOST_CHECK_EQUAL(median(dist_0_1), static_cast<RealType>(0));
    BOOST_CHECK_EQUAL(median(dist_1_3), static_cast<RealType>(1));

    BOOST_CHECK_EQUAL(mode(dist_0_1), static_cast<RealType>(0));
    BOOST_CHECK_EQUAL(mode(dist_1_3), static_cast<RealType>(1));

    BOOST_CHECK_EQUAL(mean(dist_0_1), static_cast<RealType>(0));
    BOOST_CHECK_EQUAL(mean(dist_1_3), static_cast<RealType>(1));

    BOOST_CHECK((boost::math::isinf)(variance(dist_0_1)));

    BOOST_CHECK_CLOSE(pdf(dist_0_1, static_cast<RealType>(1)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.02038159607840130388931544845552929991729709746772e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist_1_3, static_cast<RealType>(1)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.87352751452164445024482162286994868261727837966217e-1) / 3, tolerance);

    BOOST_CHECK_CLOSE(cdf(dist_0_1, static_cast<RealType>(2)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.94960170345170829219212314700442993280391160852854e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist_1_3, static_cast<RealType>(7)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.94960170345170829219212314700442993280391160852854e-1), tolerance);

    BOOST_CHECK_CLOSE(cdf(dist_0_1, quantile(dist_0_1, static_cast<RealType>(0.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 0.25), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist_1_3, quantile(dist_1_3, static_cast<RealType>(0.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 0.25), tolerance);

    BOOST_CHECK_CLOSE(cdf(dist_0_1, quantile(dist_0_1, static_cast<RealType>(0.75))), BOOST_MATH_BIG_CONSTANT(RealType, N, 0.75), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist_1_3, quantile(dist_1_3, static_cast<RealType>(0.75))), BOOST_MATH_BIG_CONSTANT(RealType, N, 0.75), tolerance);
}

BOOST_AUTO_TEST_CASE(holtsmark_pdf_fp64)
{
    do_test_holtsmark_pdf<double, 53>();
}

#ifdef __STDCPP_FLOAT64_T__
BOOST_AUTO_TEST_CASE(holtsmark_pdf_std64)
{
    do_test_holtsmark_pdf<std::float64_t, 53>();
}
#endif

#ifndef BOOST_MATH_HAS_GPU_SUPPORT
BOOST_AUTO_TEST_CASE(holtsmark_pdf_fp128)
{
    do_test_holtsmark_pdf<cpp_bin_float_quad, 113>();
}
#endif

BOOST_AUTO_TEST_CASE(holtsmark_cdf_fp64)
{
    do_test_holtsmark_cdf<double, 53>();
}

#ifdef __STDCPP_FLOAT64_T__
BOOST_AUTO_TEST_CASE(holtsmark_cdf_std64)
{
    do_test_holtsmark_cdf<std::float64_t, 53>();
}
#endif

#ifndef BOOST_MATH_HAS_GPU_SUPPORT
BOOST_AUTO_TEST_CASE(holtsmark_cdf_fp128)
{
    do_test_holtsmark_cdf<cpp_bin_float_quad, 113>();
}
#endif

BOOST_AUTO_TEST_CASE(holtsmark_ccdf_fp64)
{
    do_test_holtsmark_ccdf<double, 53>();
}

#ifdef __STDCPP_FLOAT64_T__
BOOST_AUTO_TEST_CASE(holtsmark_ccdf_std64)
{
    do_test_holtsmark_ccdf<std::float64_t, 53>();
}
#endif

#ifndef BOOST_MATH_HAS_GPU_SUPPORT
BOOST_AUTO_TEST_CASE(holtsmark_ccdf_fp128)
{
    do_test_holtsmark_ccdf<cpp_bin_float_quad, 113>();
}
#endif

BOOST_AUTO_TEST_CASE(holtsmark_quantile_nearzero_fp64)
{
    do_test_holtsmark_quantile_nearzero<double, 53>();
}

#ifdef __STDCPP_FLOAT64_T__
BOOST_AUTO_TEST_CASE(holtsmark_quantile_nearzero_std64)
{
    do_test_holtsmark_quantile_nearzero<std::float64_t, 53>();
}
#endif

#ifndef BOOST_MATH_HAS_GPU_SUPPORT
BOOST_AUTO_TEST_CASE(holtsmark_quantile_nearzero_fp128)
{
    do_test_holtsmark_quantile_nearzero<cpp_bin_float_quad, 113>();
}
#endif

BOOST_AUTO_TEST_CASE(holtsmark_quantile_lower_fp64)
{
    do_test_holtsmark_quantile_lower<double, 53>();
}

#ifdef __STDCPP_FLOAT64_T__
BOOST_AUTO_TEST_CASE(holtsmark_quantile_lower_std64)
{
    do_test_holtsmark_quantile_lower<std::float64_t, 53>();
}
#endif

#ifndef BOOST_MATH_HAS_GPU_SUPPORT
BOOST_AUTO_TEST_CASE(holtsmark_quantile_lower_fp128)
{
    do_test_holtsmark_quantile_lower<cpp_bin_float_quad, 113>();
}
#endif

BOOST_AUTO_TEST_CASE(holtsmark_quantile_upper_fp64)
{
    do_test_holtsmark_quantile_upper<double, 53>();
}

#ifdef __STDCPP_FLOAT64_T__
BOOST_AUTO_TEST_CASE(holtsmark_quantile_upper_std64)
{
    do_test_holtsmark_quantile_upper<std::float64_t, 53>();
}
#endif

#ifndef BOOST_MATH_HAS_GPU_SUPPORT
BOOST_AUTO_TEST_CASE(holtsmark_quantile_upper_fp128)
{
    do_test_holtsmark_quantile_upper<cpp_bin_float_quad, 113>();
}
#endif

BOOST_AUTO_TEST_CASE(holtsmark_locscale_fp64)
{
    do_test_holtsmark_locscale_param<double, 53>();
}

#ifdef __STDCPP_FLOAT64_T__
BOOST_AUTO_TEST_CASE(holtsmark_locscale_std64)
{
    do_test_holtsmark_locscale_param<std::float64_t, 53>();
}
#endif

#ifndef BOOST_MATH_HAS_GPU_SUPPORT
BOOST_AUTO_TEST_CASE(holtsmark_locscale_fp128)
{
    do_test_holtsmark_locscale_param<cpp_bin_float_quad, 113>();
}
#endif
