///////////////////////////////////////////////////////////////
//  Copyright Christopher Kormanyos 2002 - 2011.
//  Copyright 2011 John Maddock. Distributed under the Boost
//  Software License, Version 1.0. (See accompanying file
//  LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_
//
// This work is based on an earlier work:
// "Algorithm 910: A Portable C++ Multiple-Precision System for Special-Function Calculations",
// in ACM TOMS, {VOL 37, ISSUE 4, (February 2011)} (C) ACM, 2011. http://doi.acm.org/10.1145/1916461.1916469

#ifdef _MSC_VER
#  define _SCL_SECURE_NO_WARNINGS
#endif

#include <boost/detail/lightweight_test.hpp>
#include <boost/array.hpp>
#include "test.hpp"

#if !defined(TEST_MPF_50) && !defined(TEST_MPF) && !defined(TEST_BACKEND) && !defined(TEST_CPP_DEC_FLOAT) && !defined(TEST_MPFR) && !defined(TEST_MPFR_50) && !defined(TEST_MPFI_50) && !defined(TEST_FLOAT128) && !defined(TEST_CPP_BIN_FLOAT)
#  define TEST_MPF_50
#  define TEST_MPFI_50
#  define TEST_BACKEND
#  define TEST_CPP_DEC_FLOAT
#  define TEST_MPFR_50
#  define TEST_FLOAT128
#  define TEST_CPP_BIN_FLOAT

#ifdef _MSC_VER
#pragma message("CAUTION!!: No backend type specified so testing everything.... this will take some time!!")
#endif
#ifdef __GNUC__
#pragma warning "CAUTION!!: No backend type specified so testing everything.... this will take some time!!"
#endif

#endif

#if defined(TEST_MPF_50)
#include <boost/multiprecision/gmp.hpp>
#endif
#if defined(TEST_MPFR_50)
#include <boost/multiprecision/mpfr.hpp>
#endif
#if defined(TEST_MPFI_50)
#include <boost/multiprecision/mpfi.hpp>
#endif
#ifdef TEST_BACKEND
#include <boost/multiprecision/concepts/mp_number_archetypes.hpp>
#endif
#ifdef TEST_CPP_DEC_FLOAT
#include <boost/multiprecision/cpp_dec_float.hpp>
#endif
#ifdef TEST_FLOAT128
#include <boost/multiprecision/float128.hpp>
#endif
#ifdef TEST_CPP_BIN_FLOAT
#include <boost/multiprecision/cpp_bin_float.hpp>
#endif

template <class T>
struct has_poor_large_value_support
{
   static const bool value = false;
};
#ifdef TEST_CPP_DEC_FLOAT
template <unsigned Digits10, class ExponentType, class Allocator, boost::multiprecision::expression_template_option ExpressionTemplates>
struct has_poor_large_value_support<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<Digits10, ExponentType, Allocator>, ExpressionTemplates> >
{
   static const bool value = true;
};
#endif
#ifdef TEST_CPP_BIN_FLOAT
template<unsigned Digits, boost::multiprecision::backends::digit_base_type DigitBase, class Allocator, class Exponent, Exponent MinE, Exponent MaxE, boost::multiprecision::expression_template_option ExpressionTemplates>
struct has_poor_large_value_support<boost::multiprecision::number<boost::multiprecision::cpp_bin_float<Digits, DigitBase, Allocator, Exponent, MinE, MaxE>, ExpressionTemplates> >
{
   static const bool value = true;
};
#endif

template <class T>
void test()
{
   std::cout << "Testing type: " << typeid(T).name() << std::endl;
   static const boost::array<boost::array<const char*, 2>, 500> data =
   {{
   {{ "0", "0" }},
   {{ "0.03125", "0.03126017650125595642059651621507489147877835880393069701338513042228140662868270178810496567920444849" }},
   {{ "0.0625", "0.06258150756627501465828481747745378814263718044751609194477198396918772026898277636891147279909599223" }},
   {{ "0.09375", "0.09402562724573195576489755283678159034525247904824579838425258406126459673745015890895186868431790389" }},
   {{ "0.125", "0.1256551365751309677926782186297740007586657638922255426688666139272557034809288714758416649081949452" }},
   {{ "0.15625", "0.1575341073252716106852257741724104906661974475121544257009051769486388715993896887955248205741956572" }},
   {{ "0.1875", "0.1897286107180591328910833700730338162710122536010363055257273177325079653665703027537671734777793159" }},
   {{ "0.21875", "0.2223072805534313308722888175879995822319104918387718270273535559005416973923742249393961682186904743" }},
   {{ "0.25", "0.2553419212210362665044822364904736782042016388008226217404756502588831981346542579493165026310550454" }},
   {{ "0.28125", "0.2889081724405147260015884454642448286982056723714186732378896291489603033944207768995735063901118923" }},
   {{ "0.3125", "0.3230862443517455201183006557179619743037036164759656335249943943470835620899848486165999610265179128" }},
   {{ "0.34375", "0.3579617388480169983883959631794471406375375900948760280972554801967920011385682070576100261097824193" }},
   {{ "0.375", "0.3936265759256327582294137871012180981856966444080712949818309739116951585799388015457993878787997269" }},
   {{ "0.40625", "0.4301800474642300490203296054472752679963181652078151755042924196261397876727827800621729454863519218" }},
   {{ "0.4375", "0.467730025452391799921342706961992620413738282701114114727311802060193364671415150855360089743973883" }},
   {{ "0.46875", "0.5063943574962298120708227547071771387085972911587370796115595547436950963394141008177411206688111652" }},
   {{ "0.5", "0.5463024898437905132551794657802853832975517201797912461640913859329075105180258157151806482706562186" }},
   {{ "0.53125", "0.5875973675914432213941588631578976924522062928763012976297627647411436985181631978010052165746790249" }},
   {{ "0.5625", "0.630437673835884766852611429299775209903293946951939898251650513745196222165381994293674222073197888" }},
   {{ "0.59375", "0.6750004851442429076631779494777229128864061685897319107997999948317524157945835559284790797708856594" }},
   {{ "0.625", "0.7214844409909044199895178832795946807605892812145963657590722733756611374332889471678213489198161525" }},
   {{ "0.65625", "0.770113551344208705005983660052773240245517035179766333813336659352553100285016757206110401160028615" }},
   {{ "0.6875", "0.8211418015898941219114239653747117008753716453086877023306268412086961091026737960147423436480371207" }},
   {{ "0.71875", "0.874858760554482349539671907932155532231782257040588572549027780010892786023906162883923961511720127" }},
   {{ "0.75", "0.9315964599440724611652027565739364281886913399722189970823760758473440872606463942290222395451727347" }},
   {{ "0.78125", "0.9917378983632686802568573899299023689572109767948642785884455151680071204845003794305054375396567381" }},
   {{ "0.8125", "1.055727639411919906905928322585893795895648561782414602487481288436493656539800004691156074588329712" }},
   {{ "0.84375", "1.124085134704560944131255965236688143403009777898521429483610660496648114350313475950507899270640935" }},
   {{ "0.875", "1.197421629234347972339226617276609445975806623418086002265951386950959459751336541541255052511392263" }},
   {{ "0.90625", "1.276461828982383445402617843263224436254612963462020761243811395954456362359102470162910199798207331" }},
   {{ "0.9375", "1.362071976376228231137266640571127531383954502011184315622281795551984309995647087587519913956728759" }},
   {{ "0.96875", "1.455296662469072910283351203225910848640458285194500237036240390915811463257413817682188441906818248" }},
   {{ "1", "1.557407724654902230506974807458360173087250772381520038383946605698861397151727289555099965202242984" }},
   {{ "1.03125", "1.669970130353601668362228999287655840466243330137306547430306024592587314737242276409239980979443174" }},
   {{ "1.0625", "1.794932157265411111412512666640647365321540790157497713363548060626019826336277151761592643681562889" }},
   {{ "1.09375", "1.934751011916103924218794082343652203808222523749686394781135509079848569244264225172363092871198012" }},
   {{ "1.125", "2.092571276372179004423733981234886782259941716148720572913991515339894408382371607574079035650497628" }},
   {{ "1.15625", "2.272484060247449371554848155118626331407879143446651944395567708418983252993702557436552564850427597" }},
   {{ "1.1875", "2.479912917556758792437515574454698934001624072586467858166295252178386457324268836053055868956236971" }},
   {{ "1.21875", "2.722205296368710926927932441765877991142333599010908457349133016162381084884183783877142234320889929" }},
   {{ "1.25", "3.009569673862831288157563894386243931391637699606062181047618417194668052548525026087449023543493348" }},
   {{ "1.28125", "3.356619539863437446801254212059537717694035906934171604782430676182834617443082960603002796648697006" }},
   {{ "1.3125", "3.785038166535876194630450975067081020082611278470578710038656250838244608554256510046862645943214168" }},
   {{ "1.34375", "4.328443997051826678273102914917348524900160978527585314859991445919675557868173766988955991311583481" }},
   {{ "1.375", "5.04191525648136332110647205436422898660842030191113551461354742130497045161358084541551547909572374" }},
   {{ "1.40625", "6.02236781523945747670227618692239447637023828129226996489330674807031833318199939459376428890858723" }},
   {{ "1.4375", "7.457597366497314807346683309400119820853357043059007743950216525315136075258562571085388447049580691" }},
   {{ "1.46875", "9.765431722939984318153137075537986427219225446494026346174651506537333303940271000197727980305021618" }},
   {{ "1.5", "14.10141994717171938764608365198775644565954357723586186612326758608969627041415526864870292630944229" }},
   {{ "1.53125", "25.27361509038201021888564593774203447137999639112919943365223977030408052066444435679945642193151117" }},
   {{ "1.5625", "120.5325057225426126499709994255564804478992494692406282802019970839232698437518852292295322760051977" }},
   {{ "1.59375", "-43.55836040673973128976461522014256814059799632626649998645431997677346427573993827534701436489680689" }},
   {{ "1.625", "-18.43086276236962258434966752738448975654856844592479276671012332110139541995382323486170528839748939" }},
   {{ "1.65625", "-11.67374867215481348216513536293945130766869617358776549669176281104274968860961548954064974733991076" }},
   {{ "1.6875", "-8.529773993911653209272999783778324026196719414411590092375294867302897481134537117581472896391856125" }},
   {{ "1.71875", "-6.709482397820695376442251396240149572345307174060614879543507453718786710576833294867753884694537068" }},
   {{ "1.75", "-5.520379922509330168089935698420205500301122430036809938027371836902288281994865833387763391132527793" }},
   {{ "1.78125", "-4.681280325336429567316481395066494078522124421106252324368679821712511116596067067167862140573504461" }},
   {{ "1.8125", "-4.056414052170035671394695838412072312925972547334020477371203912084804008892977770267448199074447777" }},
   {{ "1.84375", "-3.572185661861125859870109039981697499670205624150737204067785923886464814599307447743073100872203818" }},
   {{ "1.875", "-3.185238919501229169914672833591055563068190813252958079987355697225012578404488535638528315769681186" }},
   {{ "1.90625", "-2.868371726567042158384165205507273428429609298347592604875428057959086996590505442480814626239318494" }},
   {{ "1.9375", "-2.603652912010850547260718033703880538638622594696262109914178307203938532760442947667909676035790898" }},
   {{ "1.96875", "-2.378782114044268462754557076702009514149544247914950239975234530609231791907385965883627186285245033" }},
   {{ "2", "-2.185039863261518991643306102313682543432017746227663164562955869966773747209194182319743542104728548" }},
   {{ "2.03125", "-2.016072402071886080344963601799448100479409508905552305715636690184075224937451135890591580854847922" }},
   {{ "2.0625", "-1.86713988123437020407782794323860043132451055327769468817959958079457004467338400013589662215485346" }},
   {{ "2.09375", "-1.734634103946532053863260773540976834775727590591832428183425859927779370855756294581920109583545138" }},
   {{ "2.125", "-1.615759423973460045704218998135983105718634789435135146363364502252797531256854000679335498343468236" }},
   {{ "2.15625", "-1.508315836353881219959349807787957061590021089263168744744644715237923771519750208817662219357365791" }},
   {{ "2.1875", "-1.410548011813148830578875195416796900655621273730614063607851296345238287109255775298602026828637775" }},
   {{ "2.21875", "-1.321038011883972941371242354544773427461451983076687172335288474134737150563349929598365091185906691" }},
   {{ "2.25", "-1.238627616224096475286573358483115818987811679840895892063010057942989763968416082248247538779346115" }},
   {{ "2.28125", "-1.162361144121629479886631062456510653121514437736910570208484412489182649570861593071569298439919991" }},
   {{ "2.3125", "-1.091442725473163592445273166049218471062020679322657236270267019564600575624640388147984118518979059" }},
   {{ "2.34375", "-1.025203931548868392935198625444103909279764848147237361743751176005185409221197679274034521026108332" }},
   {{ "2.375", "-0.9630789471610371325783309326887526239686440615099207880230611368370468000350460791927023591069451265" }},
   {{ "2.40625", "-0.9045853091742213054875350240709441597937968403208608690036487459318455608785453421438664490396695609" }},
   {{ "2.4375", "-0.8493088059488314622450104194218363224080665638942044768228802750401486144919082454646163741529209686" }},
   {{ "2.46875", "-0.7968915235352686521345041809817019192899787435677815043081356746793473472915009329509869865385574704" }},
   {{ "2.5", "-0.747022297238660279355352687825274557904116956883011279066593089700271855760843718388592769933778409" }},
   {{ "2.53125", "-0.699429020084844342139783831016239425344370000759297142831905446726651787263273111657622358057530982" }},
   {{ "2.5625", "-0.653872397910572369605836837404416068498526729423046632151562590225826018439381877458767572737519508" }},
   {{ "2.59375", "-0.6101408409895599924272199138429116429964861640554001463611527892708422318404310839955568952517348978" }},
   {{ "2.625", "-0.5680462555551780726648895974757705443146437970424414412380682777625097984947690751561280498460825515" }},
   {{ "2.65625", "-0.5274205529869757121326571336563206310413652365485599902702063856643631953989601868253259786432770197" }},
   {{ "2.6875", "-0.4881127351190876547388242189029165584838796506348536005759885166591812246270986452391896414818116207" }},
   {{ "2.71875", "-0.4499864448356251166356611982487791354890745233650543094554613558499691453012194865420102877343872536" }},
   {{ "2.75", "-0.412917894484932487763966873261129123736255820362062889934249403216388494265361969753200463293305801" }},
   {{ "2.78125", "-0.3767941025635464453019403649599866905758314392515266200296815335892123350214817069316692446879993786" }},
   {{ "2.8125", "-0.3415113829599005939054709337126887266408130303329754956325710946559071503247227755879401227074384719" }},
   {{ "2.84375", "-0.3069740418053874743538801719778118186539092328300496832334808000582741898558068313989599567333495833" }},
   {{ "2.875", "-0.2730932453907306446782089914481699536905769611886147850801712661656164014954781669815730570254890551" }},
   {{ "2.90625", "-0.2397860292133634315864717364763276794949749917725083450148739709007738169658284872978100162537831057" }},
   {{ "2.9375", "-0.2069744234334290889140606202259876249518274239071734165293810142417092926391409078539182667795178826" }},
   {{ "2.96875", "-0.1745846741378176862031217701953393007964354401947395498383109594878301292911980508212117939922029155" }},
   {{ "3", "-0.1425465430742778052956354105339134932260922849018046476332389766888585952215385380591060583477669114" }},
   {{ "3.03125", "-0.1107926710961848652307179061543482824777047419183290635786540267776792824053626901819122659270065422" }},
   {{ "3.0625", "-0.07925799258597813975964847855303205226643624625818783981886842299834648957527837719024324414633813629" }},
   {{ "3.09375", "-0.04787918970239737242913390208792864526960233706882073548585416320272138422285616307162127608218497512" }},
   {{ "3.125", "-0.01659417649935840190191646601956655481659118246051776036792478758448298530865094236374141834995963483" }},
   {{ "3.15625", "0.01465839615112051363829193218735346545420611046761168233490628016451488745447998876530695315188732072" }},
   {{ "3.1875", "0.04593962329265888212191655478619122278727029224118607262776824432506068644004151566997697327162767079" }},
   {{ "3.21875", "0.07731082437922182842323045510047455560710445548270932232603347976675416898602720984389229982663062937" }},
   {{ "3.25", "0.1088340255133297195139603974000083015109646348076838424061656004036607221464457317348103204356071731" }},
   {{ "3.28125", "0.1405724535178196442425946789234394740490046721659763531803771463602023716095051057689530418533761391" }},
   {{ "3.3125", "0.172591050272744891901962315250310519786980119755314731828011976218754514667360156546007027602360458" }},
   {{ "3.34375", "0.2049570163396922924222670027465798026391019465616329916778651389384367221289434591672530287680469817" }},
   {{ "3.375", "0.237740393762143806988724415397077124401764815151774813277613941354927085310634989951327775643776322" }},
   {{ "3.40625", "0.2710146991120596316784994914008470605846614604634004480958302230976152416477575814862222371539690966" }},
   {{ "3.4375", "0.3048576194048116313509686061976383012732471442214687629070980813100980504220010433668547826451536192" }},
   {{ "3.46875", "0.33935178550149794902043849733017382894738729159656755127388089028458110314950118486374283672385814" }},
   {{ "3.5", "0.3745856401585946663305125799891473884508822842892592306930231225774355972615263725457922224498281269" }},
   {{ "3.53125", "0.4106544211005646795132687105613188412302653916099336518744517138690111552283413041415216571077234329" }},
   {{ "3.5625", "0.447661283553583556404846819303519593092219301719424669894622854440119111664920988209774876249304265" }},
   {{ "3.59375", "0.4857185918149514991806435613800961704649349211376530235980832426061445834319868210377502054789684188" }},
   {{ "3.625", "0.5249494159434426029201266393240939090334746308513972623727472919160131375298663507047002581644827389" }},
   {{ "3.65625", "0.5654892779403682511970427653303379875705726194565986095362182202060326386665976843463927284690694492" }},
   {{ "3.6875", "0.6074882023842700195137806696254362966726637333828709216522375901698082716779542882925451551904376524" }},
   {{ "3.71875", "0.6511131401046877694771154604832169765858772246556092408841970307496747807017677126165226354908150626" }},
   {{ "3.75", "0.6965508511114601330600221624003410002384063251948773727536766307711270568580190864635818668189622464" }},
   {{ "3.78125", "0.7440113559767865266278722337979770721544668975249127911732590346704528413537145792818180362367062195" }},
   {{ "3.8125", "0.7937320950520845038640709755691971987193904745863880210276991068348251595048327258835950484424829585" }},
   {{ "3.84375", "0.8459829748791345210684014414635237310065611239815471815195561711011715779741435605360151278986068963" }},
   {{ "3.875", "0.9010725345768489830226429310383860542057548011724428357500976339946729939388193110037885928433616767" }},
   {{ "3.90625", "0.9593555370605378999830231072930600308342084750238673386763085477722047681353378531697620720257631096" }},
   {{ "3.9375", "1.021242388200347736200680021836703924511025898807046899043703227626470466798902950286611342705663006" }},
   {{ "3.96875", "1.087210922446937924973630130633474302114867753655250594086911814333271884554141747835703519160797518" }},
   {{ "4", "1.157821282349577583137342418267323923119762767367142130084857189358985762063503791325897570132020188" }},
   {{ "4.03125", "1.233734886297294089608083483431716940428204649552146125435266924766182484146724296986257289579785258" }},
   {{ "4.0625", "1.315738861236497278595002752513023767487817470172494220905161864739955592222079472915707284820672627" }},
   {{ "4.09375", "1.404777873409928816791398747372142291632709052730067982243952454765008272213717678329031865335794425" }},
   {{ "4.125", "1.501996112821481350670183703711495663471610930233216088609964712367465201389736354667815373825067405" }},
   {{ "4.15625", "1.608793425833300449400383022559449738336897468383397721431496100147808519893555398680197430417094739" }},
   {{ "4.1875", "1.726901492741173774258474522465444605840834193796631414995320087774658896048364103982815258692096685" }},
   {{ "4.21875", "1.858488933532261262387603943116424040560957193049648082245398381173590809878153982869279424951736518" }},
   {{ "4.25", "2.006309027858059435312199733843025004203991000201116744630971025635743461495678364523854925961565933" }},
   {{ "4.28125", "2.173911671176453464233801926617652208987805422324233358984523813313677124319401603031463553334586795" }},
   {{ "4.3125", "2.365954706436810886261405424430153147538433069393544616996159851851045098884294543485330531184033036" }},
   {{ "4.34375", "2.58867360110912318951821732893635523217217883697807084314676743075156168098379747192932679404932885" }},
   {{ "4.375", "2.850612135014197347464512007371469511624647336547380863597026522958389351638925329325864211361675359" }},
   {{ "4.40625", "3.163800603288976619542297558662099459127776345948257172928079751633780670148333702513937775996967237" }},
   {{ "4.4375", "3.545737656638732611056781822478653525618506322500845353570490033964184425114602742381228738013800916" }},
   {{ "4.46875", "4.022897320897881433471367965265766624497222626661161034375557752511167850308209715183529923181209519" }},
   {{ "4.5", "4.63733205455118446831908669495962673002033167604849908135592490784364853630436659035964429989580423" }},
   {{ "4.53125", "5.460110732358225081268571201584576241967507554680233568022847841671382048904153089722943992228366072" }},
   {{ "4.5625", "6.621566538917184779658822397475662809951450423939232173471067230932326374886115835513736745573657993" }},
   {{ "4.59375", "8.389349373172668043480112406855801804399985563752529469857319424831372393351619823015010870227400011" }},
   {{ "4.625", "11.41394586984414667039249686700424655784704375419079662399301751232175345550471908433956600220150528" }},
   {{ "4.65625", "17.794217921195291872116110930917322871062931030345133311457707064544221956249568351463091932558272" }},
   {{ "4.6875", "40.17012705640036449304190879290789586501901676251470634007061519017512737181345480527620292167336129" }},
   {{ "4.71875", "-157.2053810419425516610333761767147984621064081691004376958977511691902151494982722512124394464246881" }},
   {{ "4.75", "-26.57541423100021444925624133456621507049828719711987551420375127874685894467704404079281767156965171" }},
   {{ "4.78125", "-14.49904303397407169401387138739025359024482851876427547784737429808056788404653273313794074954104547" }},
   {{ "4.8125", "-9.955517692663199659497301146983549868138803040917546840671025802878897403111578981570237166977305366" }},
   {{ "4.84375", "-7.568770928466597669950125714728032038595443682400782384350047230172533622281584347829485301888539665" }},
   {{ "4.875", "-6.095345265174791895543529207308251915058396610943153120402963827367536675687235397631104150517714925" }},
   {{ "4.90625", "-5.093551747761417849838605582864480592041118085930024194611183352044275362666663197560756285563557591" }},
   {{ "4.9375", "-4.366960809832423113382187614785174399552256061595826665894310062349100292377458713017173941901099366" }},
   {{ "4.96875", "-3.814918596744426621193354615307784649480559936141816068738250826073519922812423391628708426424484949" }},
   {{ "5", "-3.380515006246585636982705879447343908709569208285459877176820980918346010201206056831791856009410428" }},
   {{ "5.03125", "-3.029148102302771164961521436423971160566131129828714146460894246137584114582867735365596418294127015" }},
   {{ "5.0625", "-2.73856823245603647865430226656274757425295307326689315789016177502179223572750955725123296691662252" }},
   {{ "5.09375", "-2.493817050876063510228483889992811590654110757263765873027034424090589388194896708294715609776449821" }},
   {{ "5.125", "-2.284466361999853132376034939699606124682746442696980078901017290742647881722505799438755300967817548" }},
   {{ "5.15625", "-2.103023353734157432798966856819893312894278382156270431204069828145019096032929826221618815522735662" }},
   {{ "5.1875", "-1.94396519241478443283285298645374010314101176847753574637302261542129293949555826361500398886874374" }},
   {{ "5.21875", "-1.803131092461690600361529359360266463729989122374270260332052248390551413451160551386405337128358437" }},
   {{ "5.25", "-1.677326406624389127684834446781355734878514221167662440011355600468562011653802826535309338734910053" }},
   {{ "5.28125", "-1.564057206075927617852462348582481289639348906743456817576975327632362277790709160010002260498590699" }},
   {{ "5.3125", "-1.461347784441303570775301256747389266430752496556055422309798378517676612642129660840051802732712867" }},
   {{ "5.34375", "-1.36761235448791393113713404156603997628029078648313063442112198771680848353561063900933679869893461" }},
   {{ "5.375", "-1.281563046345464215429873432637423222317866003116950751512613373376845314748305445490135781753417754" }},
   {{ "5.40625", "-1.202142762311081825673780483344209940977316276937937096782702116019344314429792428912212274465011549" }},
   {{ "5.4375", "-1.128475389186388730418668294914420100814289081230956940736485357975544285270667364906582995184004442" }},
   {{ "5.46875", "-1.059828347715565927266867602143137996921061876523369816266086983272908895834550263458873978282357056" }},
   {{ "5.5", "-0.9955840522138850177019161740702867286905993509040201421089147368547994233084178348523042915084083077" }},
   {{ "5.53125", "-0.9352178996735373313249149491276249977722839556692437323635652451207536878974422360418217978053551773" }},
   {{ "5.5625", "-0.8782811077026627420028846442823333510762218192932291958379466783448865095622083395150126065512344446" }},
   {{ "5.59375", "-0.8243871973186229222115460053416379931615277523370335469512772188350449840600638037662602397836368723" }},
   {{ "5.625", "-0.7732012463758734614771474685457909254527078194270700631060606366411896649393019485319264721701600447" }},
   {{ "5.65625", "-0.7244312708999757737148502388902211557937330882505616499576249426224616653330080051888518816616123148" }},
   {{ "5.6875", "-0.6778212563139696041943536024958433042316769928076303165171422834039846729764625273526488517789737417" }},
   {{ "5.71875", "-0.6331454792161440424929262529011204753164261484247750443857435235991899745283883164285814469564729784" }},
   {{ "5.75", "-0.5902038468644901054170195032663081108054249240846444267071062035473153343668376567479794165486184472" }},
   {{ "5.78125", "-0.5488180452487470465583585186996023704202388711137425402809495353782052794835324270116435144444853907" }},
   {{ "5.8125", "-0.5088283340526268433264568968479225952416460593340160732760808861658301198202358775409378607410253813" }},
   {{ "5.84375", "-0.4700908624280941936361204610007198415053847982327139590868905985009927783431252538270334618349713453" }},
   {{ "5.875", "-0.4324754064907757695440115128522408464584984965503283137576460100497915888775090560945910034482914049" }},
   {{ "5.90625", "-0.3958634500577715728387322802090283590444850041488349012771936138475077409268246305105488709941047507" }},
   {{ "5.9375", "-0.3601465460097104261765989532960199175446047549989388477273964911510226366930877544365028188123550406" }},
   {{ "5.96875", "-0.3252249079465421281971924062604461228563045080514509080291865924060718896122952297947720584304400305" }},
   {{ "6", "-0.2910061913847491570536995888681755428311555709123391316088271933046559131122021519996248059475229616" }},
   {{ "6.03125", "-0.2574044312501190948926553260844066507272665964422420930835303888833069876363053415593229084989769505" }},
   {{ "6.0625", "-0.2243391083295019990486082083153394557139168759436924449745473819036543999446390847474833972561431122" }},
   {{ "6.09375", "-0.1917343220126000636733861789852210342976612418141698349764104452472624060435608559813294673409175422" }},
   {{ "6.125", "-0.1595180503486741129487427411798597329274963598657430717585348638114328150730071113710855848882228728" }},
   {{ "6.15625", "-0.1276214813658537291522157530092700353508309312306960197241502999632222071779029887930485312029959035" }},
   {{ "6.1875", "-0.09597840190639155007190320592294119000151190572960121328263061886000450929267149036536453259308092125" }},
   {{ "6.21875", "-0.06452463203653075216342354397349405171379838323909874171208799994716724111404978040328803565744248937" }},
   {{ "6.25", "-0.03319749448301144438419631194317525127766038457693080701966081495473675499225984399367237591921526005" }},
   {{ "6.28125", "-0.001935309595765553843691829525274435788517976054625627169224279956186864996201309127614969112751901202" }},
   {{ "6.3125", "0.02932309291350979792274199376680897224174579057648744678958323756547521586508413864276599103614351545" }},
   {{ "6.34375", "0.06063885372047505801221326406404454670246033034497805406862823133882341608599003383282431176702261307" }},
   {{ "6.375", "0.09207356314350050835492397132807778236841887606137324897895572241559097713132233298612222140840431269" }},
   {{ "6.40625", "0.1236897479096077296070838372979705928368188283231838414804934929274025398571452124588393065510099759" }},
   {{ "6.4375", "0.1555513736514298191439929689618245053471522825437403514671018199418895089031472964835239198291348668" }},
   {{ "6.46875", "0.1877243718114725376401948692378642961122451437059168414418744613043681130520375192781612678122611322" }},
   {{ "6.5", "0.220277200345896811825735903245910395013134783433298671312565971040207859095962733489570927809983752" }},
   {{ "6.53125", "0.253281448628424489171015810263074415662213222146436493340123226398154356922736152388141063544543841" }},
   {{ "6.5625", "0.2868124983022992902396435016696204994189572656467726352599962917225918966603261177449285587256173481" }},
   {{ "6.59375", "0.3209502535782458942725336084114371332732723992168430541003499245591367871623197594684565991267905584" }},
   {{ "6.625", "0.3557799567143647005420612267605866339037398845912245431900523492992941964222774204935357299525662811" }},
   {{ "6.65625", "0.3913931072527633586303310805574337028074307951034635414359349547707586122479246892158861767238081689" }},
   {{ "6.6875", "0.4278885071761893999613016206982314187458164558702387841850210320754507932238151571774989545334487634" }},
   {{ "6.71875", "0.4653734586819128141559933341200671578345456984030117446615279672146457039797505409854792515433603876" }},
   {{ "6.75", "0.5039651470083475548132524571742890116543823430974166616625221219450482100216791892806202009417424483" }},
   {{ "6.78125", "0.5437922480355643338730870787135605883829637094412174029296025101763018004204025974330398781596165429" }},
   {{ "6.8125", "0.5849968096716556782374159174641655681779685895667304240747256104820508784969531169383146463204412289" }},
   {{ "6.84375", "0.6277364679474334892591612082171255002710958195004825881252644517358490289176441185242244713866378973" }},
   {{ "6.875", "0.6721870741034611638659173114401257239025537597591200195749563598359978601937626091399638175089362186" }},
   {{ "6.90625", "0.7185458288995548400076077740862700950460682676000352992576300205153430366069275776925317595307700138" }},
   {{ "6.9375", "0.7670350464671162556530262845461572442649216262010355745804068537455730294466119988902478588983034749" }},
   {{ "6.96875", "0.8179067044246350052260597321820256016080683284701549230268748786223049718597359248253027839270497755" }},
   {{ "7", "0.8714479827243187364564508896003135663222034245984200644480154523301674476556679351044748556811165168" }},
   {{ "7.03125", "0.927988055100702558366453168987743089267307369461721117972360539733706484657902890137954063374826404" }},
   {{ "7.0625", "0.9879064802291251594676053789880311203133455080943572829265181459625625927973428370265184881799942024" }},
   {{ "7.09375", "1.0516436537462216418830263449985522705286903110788757629051396054699693751826899237943523838887297" }},
   {{ "7.125", "1.119713940340697920929231941973033073135897434271449197914919579212768620587877263960959541708252549" }},
   {{ "7.15625", "1.192722326901007588684149485242155872646533354464514409876904886587006629927655098723914331692457874" }},
   {{ "7.1875", "1.271385753083766195102512866837447084316821755807112034055075019301218550802923593730276046614253933" }},
   {{ "7.21875", "1.356560730687106090381626490498188582130942795512557181635747233593968595718598671759051546605979157" }},
   {{ "7.25", "1.449279530126825400966804908182545262594041927922324311633865090981669827371184783228521282728877249" }},
   {{ "7.28125", "1.550798206734497978345598394969907597227586253061565831974177990152095709794984147425013361851865459" }},
   {{ "7.3125", "1.662661250535958572647253880444919849918733690439628459569418630367100238120581497993662350014826086" }},
   {{ "7.34375", "1.78678998697496418766617671089138013384203233815466866101103271149031802024685694556034074365409757" }},
   {{ "7.375", "1.925605576103531021088436315604614476223254075602410795428544750301783382496712246458936951616523844" }},
   {{ "7.40625", "2.082203514638078116771326575602059859463528643293617740928645387731580498162533103205305011512631601" }},
   {{ "7.4375", "2.26060669237264261355866723509904048590096081733761353611694012356866351200745710198451308329107618" }},
   {{ "7.46875", "2.466141609729379985621243345839389610788373993111041429799962056660510586368685947727836988290301937" }},
   {{ "7.5", "2.706013866772690776762802272387995008883446304522264964682388787620940583840910641036913840080466461" }},
   {{ "7.53125", "2.990217991868365275378986251269668199642886675694545360724616292050097614350992705530127905271866877" }},
   {{ "7.5625", "3.333032524336291185691514966848310245587002670642998146766838315216031204958906362527023258344720302" }},
   {{ "7.59375", "3.755592314437586774843968169016218401146579356484719112952387050721790775719524338750656820475167322" }},
   {{ "7.625", "4.290567124941375634141691032410770755182292184491365711713765284134592087921445364428118108025228059" }},
   {{ "7.65625", "4.991276730757292357917427430609014256293569739250152073504758948209194194425629988371488577185989884" }},
   {{ "7.6875", "5.9510718925579227833281678276540391546652969268664831508775318977823975271243707799244732578159289" }},
   {{ "7.71875", "7.349587230185289251288167334999299680627536804325241700547934306594013397091284138506685042660382893" }},
   {{ "7.75", "9.582397403456211211397313849974932847186773522851876679318331037683816539880381009086487008735246101" }},
   {{ "7.78125", "13.72492306901017608779850845108579464520662673186624419937919970195919428249551440291202224739747059" }},
   {{ "7.8125", "24.09322543772519084215097143980037097594901385135011887012038200799428585198794379628997732285582328" }},
   {{ "7.84375", "97.73268930511967729279698949083284985794762903146492273796908122393941623833579266193628175149777252" }},
   {{ "7.875", "-47.57043136522765090736927872199715977332175052521738553413789183297556632881030788665564177584345509" }},
   {{ "7.90625", "-19.11460505409424788533123395282424237931209756369617281068252559403871558997768678144456749005346186" }},
   {{ "7.9375", "-11.94556191307435573940630678240021478719574633441063345522218692801937432207824662871635357235633214" }},
   {{ "7.96875", "-8.674912622365588398920385347547421582720453172032529527100491584226967033946983107374195473719464544" }},
   {{ "8", "-6.799711455220378699925262759608633364881412661473712848121891484119304982720857546088989511220313601" }},
   {{ "8.03125", "-5.581950808689684669549294209348398629337304002995630540331278821921778288507014685935916701606576575" }},
   {{ "8.0625", "-4.726032195156837427410372612416284237028321048638455837876262548466898310633589798461547191163836986" }},
   {{ "8.09375", "-4.090461187975555242897409987452358342061343516721018283559611963305885456193756951211377465919516297" }},
   {{ "8.125", "-3.599001897974260972151248247566727216299014591900000335282979733754229851045581086180509810930082767" }},
   {{ "8.15625", "-3.206943184854794792051938572955176223102804950834430935020353335808782541779058955201736266932317623" }},
   {{ "8.1875", "-2.886329592416760158849088002905458575851623094189615497250550481681316568867901585210424096370674517" }},
   {{ "8.21875", "-2.618783945156319374614290552008768448544289531418583013216952580114253625700948540561589098351543348" }},
   {{ "8.25", "-2.391728174447109499408796092049518207948088283366738019362364536579358984891061627461448588900266354" }},
   {{ "8.28125", "-2.196262571237231366422169733787602194377870425422442983742466664711276453593062387209663908411122531" }},
   {{ "8.3125", "-2.025912262702788461190762240682190168816613310552683831653429045356666462021227127736084042613847923" }},
   {{ "8.34375", "-1.875853575963005310284960792984729880727224645450131943927210730483841090952411053279594957173517387" }},
   {{ "8.375", "-1.74241880761627240539700340437602803932808901813845968268891377391077356174950266498464542879686652" }},
   {{ "8.40625", "-1.622769124048113033597644536164719722449235338830292053581380926565132897863391794369515824501033615" }},
   {{ "8.4375", "-1.514672563181530134110926429438194143845293739750523302549165608155316405124742878421871789886842147" }},
   {{ "8.46875", "-1.416349739641873214089748540260482127074454913435673678582846741597865860350934848935327297727726336" }},
   {{ "8.5", "-1.326364327785606923837177509628989183019847616858933039728626565320594601352898256466729337047543442" }},
   {{ "8.53125", "-1.243543859514378322699255610900405730249411506966951588519354941299030591219071989214839064576899506" }},
   {{ "8.5625", "-1.166921477042600182902656425322990210134485019242760863805503078706720819497128002394896106356209544" }},
   {{ "8.59375", "-1.095692443843841888393214442111438580869887527193049760552789795233066505667689248095554577112371625" }},
   {{ "8.625", "-1.029181226242794530622671562005776602295969084821805226400002870767989174122034536280082461534287795" }},
   {{ "8.65625", "-0.9668162629797675117200288525867759956705289684684092618655305272884560415249588622588640534660859208" }},
   {{ "8.6875", "-0.9081104046373177832576347663891532295331575111187881417687389316634049160994876797510769206182893432" }},
   {{ "8.71875", "-0.8526455881926773939956486059670750827771160223765816890633669305249823793389660138647216198961396045" }},
   {{ "8.75", "-0.8000607122131663748996935033554250873516322431753295881939448427095672520914736920285067785494755945" }},
   {{ "8.78125", "-0.7500419570576314664068876238355166376078921127752475362685197014336164568978380363973382424380027378" }},
   {{ "8.8125", "-0.7023149914660190721137439943663886714896255005161995584377885528996116609571672651781263426423264383" }},
   {{ "8.84375", "-0.6566386479456336287953742502996364274330999062289317807304964691314686325574451710087769117377447232" }},
   {{ "8.875", "-0.6127997515327654285376783646363991279819055057055158641281938301880811035156798875165049539694804864" }},
   {{ "8.90625", "-0.5706088613603129182915651449940477497513792905530365057563881633464714239146849295760180740616051507" }},
   {{ "8.9375", "-0.529896739873168003806242065594283411489762262385879653776325422716635759568492302090915293632301029" }},
   {{ "8.96875", "-0.4905114059515517049978410600006085544703557897386518199219966407731976599471950962402695227799723485" }},
   {{ "9", "-0.4523156594418098405903708757987855343087169256619843288072463344895615552699928891423222740611696987" }},
   {{ "9.03125", "-0.4151849883541607117552210974950947708368354992200219503945689498048994961045141832240492234717107402" }},
   {{ "9.0625", "-0.3790057881989822857530457874776947516302198077664862408128815699029163107706671425087481058924528226" }},
   {{ "9.09375", "-0.3436738369929040836550521796260295565231979408984456062837271050291411394204206333260572683795806063" }},
   {{ "9.125", "-0.3090929803907614767471346771315481283472474354946281905565967857578879461207205702332049986857635056" }},
   {{ "9.15625", "-0.2751739899376987485248300510983424723605373451217689431344224438486165931202115441514245211478398115" }},
   {{ "9.1875", "-0.2418335641420615678638155264698574025131647248333694816193249713183100110082265076470052022707012095" }},
   {{ "9.21875", "-0.2089934473583106794872121428783074116994340488341375724854578952783525682316496422613143306038638671" }},
   {{ "9.25", "-0.1765796456511299761112751840598406521389487242985432214413528930195762314982591148356765022144220392" }},
   {{ "9.28125", "-0.144521722122150591814619269577170465800891448238859437753351534510505894724788264050561053752147101" }},
   {{ "9.3125", "-0.1127521567973819422597203631074129348120757432615503831725905368424112610292538623181492998345439551" }},
   {{ "9.34375", "-0.08120575823178342176032872038448411443172784451696952348584885928982452335749552547386985037617069992" }},
   {{ "9.375", "-0.0498191155899860392003104787951054631853528180570175470879940653716436714322811615959161544657494618" }},
   {{ "9.40625", "-0.01853008118625405320915271202115144102822290582324439033642733950763784110773883754760179758258022532" }},
   {{ "9.4375", "0.01272272563027105977935573889462983744829956962204063464126832673322950722546841176709360279348538747" }},
   {{ "9.46875", "0.04400040173584953885682440835958114627885199630660893827828903912727543614728912707303196060753625141" }},
   {{ "9.5", "0.07536423875739405740654014205323665789342966267450384287846707728875924157814476147945907817392192786" }},
   {{ "9.53125", "0.1068762048461025163674352287858142449224256724379234993543813605972685100190359293029737215855453465" }},
   {{ "9.5625", "0.1385994377761661718602480104778901941178589868631124656851160671189176086579048746331765092114253163" }},
   {{ "9.59375", "0.1705987577719601866335639057437200427987763331813918550294340474134857881430332967127222805799022305" }},
   {{ "9.625", "0.2029412090417411123603662227278784038776298381618810811646084653751029723759416476460088872408054991" }},
   {{ "9.65625", "0.2356966398441474632357392833080328068352054698543206659897268100728097017667724684606559049124154157" }},
   {{ "9.6875", "0.268938332074403309291166325486527607211725274017137932139784676368590535825858340593417854926297648" }},
   {{ "9.71875", "0.3027436928841933821970886411755473788376234496119460628913545818204169198969541686968930178216130025" }},
   {{ "9.75", "0.3371950228159297335621484819773554194812863370480952763499236064163820830033268339692324715894157813" }},
   {{ "9.78125", "0.3723803774359345699506506000690612357176932379964207743390433297005778841484733220050565284302113607" }},
   {{ "9.8125", "0.4083945426204256871181400863951603330453056915705065091908373104020696107020950522244595815242992" }},
   {{ "9.84375", "0.4453401476523343154827952361952568691359033281221121303793343966901885521315031273219697042193257942" }},
   {{ "9.875", "0.4833289453490298299990038076226424305503958067355696981733291906588949898848660168661363237230401676" }},
   {{ "9.90625", "0.5224832948561672053074357426518017813969502851835052528668887798173320745102979706674984553747785107" }},
   {{ "9.9375", "0.5629378909036186388354128640753191217895638800506841967942363059454826038558876839732857927717227769" }},
   {{ "9.96875", "0.6048417937509008993401606503437317082375426212574423350794174057182764889826011397690814642006452007" }},
   {{ "10", "0.6483608274590866712591249330098086768168743429837249756336279673958556003746239008717172062971522862" }},
   {{ "10.03125", "0.6936804314743542712398847790886887683057558913351293442530262309817850224201884209136086813403394689" }},
   {{ "10.0625", "0.7410090731101333158449425961247175839017373828496178608167740610459452518219259199024083200316084101" }},
   {{ "10.09375", "0.7905823581871925973468103245168030277845433796367670128316098222165598426689171297882306893568062531" }},
   {{ "10.125", "0.8426680163689236497241316220032744251657095955743734411730267063148126128540903336099720150350195161" }},
   {{ "10.15625", "0.8975719901861101829801016952033465293839624897512338971033212391126217276476390113488747550991133784" }},
   {{ "10.1875", "0.9556459274758633915064588099748061258315512098776637126437164255246444193412583209025710708854064641" }},
   {{ "10.21875", "1.017296473311965646505591388794303117791831049823203228782533350726187433458132860834038283268836837" }},
   {{ "10.25", "1.082996890215945275135930985916637983213229101767681562664207604721345006616414244816945101012068962" }},
   {{ "10.28125", "1.153301720414501166084164165377670663086784905305525053523750216145961391216299317767459210812275763" }},
   {{ "10.3125", "1.228865465058231603544476514726969780448778330773071834149679410133713041325810373844792660869193974" }},
   {{ "10.34375", "1.310466629156747359552320115718026861262813059499884349607868463460206624928464716803324337065850131" }},
   {{ "10.375", "1.399039024262115717131604647733708995170160567832938194238903453989697561972061091597355404503662754" }},
   {{ "10.40625", "1.49571302349195916111981867105861656396166182988283681852423319036111614206596318006249059781485835" }},
   {{ "10.4375", "1.601870670512520509271426433671643695090231605946938642547694243879213679699085414650549829075718711" }},
   {{ "10.46875", "1.719220395521651499721208794808194478968865549194510942718291121399341439043920644901884496910696737" }},
   {{ "10.5", "1.849899993421927279907583573513910565419553685959044018741794327382830123593588324994882461965573398" }},
   {{ "10.53125", "1.996621179417850643664093294309064624983038139124591902589207528735600835709855269195187933895652101" }},
   {{ "10.5625", "2.162876723677546330469959787105963848827689014945066557592989408260164364548287694286197662196239537" }},
   {{ "10.59375", "2.3532442330793790868766441879841993750130572630209439743618231460831573709883106394866434440837641" }},
   {{ "10.625", "2.573843631271674362526537866893410697197766596999920083319154750577002457483161716337803596844925106" }},
   {{ "10.65625", "2.833047421190427952095570132770964879184235914846751261896726812931990128759669900544999535061769791" }},
   {{ "10.6875", "3.142623220176844496098570587446827716363816198180813628746017544330426339542323699691642256030772053" }},
   {{ "10.71875", "3.519650155321593637030342256680131753850278370827948246193900683195030354927519452141570420409057437" }},
   {{ "10.75", "3.989898450288877172272878730129464264004901452428968181149190336688176169073686050444717716156454253" }},
   {{ "10.78125", "4.594165609870243515783279467995060486532380531283432068617070247011706442921648714627407953983920726" }},
   {{ "10.8125", "5.401101953070107889048337369479695723086844105989269027204537226522869955593746957868263836003479647" }},
   {{ "10.84375", "6.535875415517406207251199308597767679104658091064327476058394143828942638073632382427758536504917985" }},
   {{ "10.875", "8.253411766339305537723460693388679535974907155823982004597806731192500887738499761977958603336992426" }},
   {{ "10.90625", "11.16537284500979684723686544686940420453225367789354062936636940962591367864691678374577954865828186" }},
   {{ "10.9375", "17.1999620041711094734838984205883317149769089588687370821690524219299315809266573264745104659869829" }},
   {{ "10.96875", "37.27070620721793879698276661020742630154681506539144657714408230209752760393928366701053772541231071" }},
   {{ "11", "-225.9508464541951420257954832034531539516575098851311946310945881123661134104031654779989090875181672" }},
   {{ "11.03125", "-28.01838123473997016830887833476199550560697066011986939559933525309909705318537405182199378012840027" }},
   {{ "11.0625", "-14.91962506867234581945151895985710289930225918712794589354339983163930894702818997439864076091931126" }},
   {{ "11.09375", "-10.15307233756673521728139815893864848321827456537331024001764086854554203725293661816927668294489745" }},
   {{ "11.125", "-7.683249828520493578856827081745572284486853920128557099054138395954224416767780846216824234128741871" }},
   {{ "11.15625", "-6.170065007339837672004969274368706522165797321914087418607341946608032746141308987068347258026539261" }},
   {{ "11.1875", "-5.146216397928971013498356330072590228022047015189051411856482418633925739244947714746552179251733944" }},
   {{ "11.21875", "-4.406134215367990783634732996136714284665546485639321948525683144032506137127272195504713779988841399" }},
   {{ "11.25", "-3.845243526046256589195292015214843119217023542577367989034898325348687250589462590349993156554307721" }},
   {{ "11.28125", "-3.404725196340913894767892653680241020178288686778598191582702368768260415894176917496336011269246961" }},
   {{ "11.3125", "-3.048957435167265586183006322737827462691340358802500561060757089057213237102895081676386924312658609" }},
   {{ "11.34375", "-2.755105539085522466956356685286240136460553349428076133380205505661678601069631697225730693586354901" }},
   {{ "11.375", "-2.507856046343293846165268336837981024254374791359792671451310995149841786950224129840033693800933455" }},
   {{ "11.40625", "-2.296555085357014688557714000971300925194148838031073606034513705752180239656706450912207637184212202" }},
   {{ "11.4375", "-2.113560858723099424307258719446955475134681339177645144130883477669849838830392714536871823793018366" }},
   {{ "11.46875", "-1.953248965442351814769641861822757418968218808226145477990020040801933787597505710461744957513141637" }},
   {{ "11.5", "-1.811387450326824859772397320474032833639713335926721521705076560469479148263546424971811321096648723" }},
   {{ "11.53125", "-1.684730597525019137697751820797269016453098562968854704274935631922618587469362028272636692210544871" }},
   {{ "11.5625", "-1.570747064811472615228088301480862011854346986220267355439332676174065617757164643849543074172144195" }},
   {{ "11.59375", "-1.467433230574720919411941987695161181959171583863783447843463212366085434611122065381627004242638947" }},
   {{ "11.625", "-1.373182138457896453349468046455100980035443259553659899722906361390876088856423969181083376455705666" }},
   {{ "11.65625", "-1.286689630920366616376314221177221139006716016538511655589760746532540573609680625603641074608188434" }},
   {{ "11.6875", "-1.206885914221201720054271911345572193315238450358663840449617121968073512366326525649980278803416278" }},
   {{ "11.71875", "-1.132884861825656481798873135046969210301345541577298019300751660619960162890235612177970957925478595" }},
   {{ "11.75", "-1.063945912486004409710559363045985530861938320222047288703751234746510874929602501339045816423036172" }},
   {{ "11.78125", "-0.9994450559333915167440624304322053019403928948191033206551515159313846021422418864273459645373449009" }},
   {{ "11.8125", "-0.9388524723222944773205643428254523492068640072627707382490811032852295370278108292005146916272238994" }},
   {{ "11.84375", "-0.881715108900528303919177723946502416645736801861050134491985441499235913287446780705353461545417442" }},
   {{ "11.875", "-0.8276429652929236471146892695710241706925495072977152657326004944101623406458452837545349486881138268" }},
   {{ "11.90625", "-0.7762981960095720820932228114020179373213795483270746275212139165301756520252573166217954465331159458" }},
   {{ "11.9375", "-0.7273863753138267917778418153001732955258816548283145725251801363629114248343367970028832122142992284" }},
   {{ "11.96875", "-0.6806494377457210277597470928336025589584620324058462321470850608946457200573674297093242265437051282" }},
   {{ "12", "-0.6358599286615807924609379429390076061509587586686860914514606814656551214484123291399643068894751555" }},
   {{ "12.03125", "-0.5928162873294508906897144974924364212100987037074180867630829726623330818215300743873013630492129075" }},
   {{ "12.0625", "-0.5513389500434250704976885869765975030435162971228425993732710884431519725994082993307502135366704687" }},
   {{ "12.09375", "-0.5112671090040856778211886692327830312529713244160725823950417285454010972749346501560988873183882094" }},
   {{ "12.125", "-0.472455998959126666328373026314261347130558625459848565935809588092219192527888607849005967299526205" }},
   {{ "12.15625", "-0.4347746110466813389220145482909141465629503223075144609806539759358566301078783883593960376790796208" }},
   {{ "12.1875", "-0.3981037542382981122358374348325646108510087325306228562334315140437947730049798142715674391074029588" }},
   {{ "12.21875", "-0.3623344008952777173267729097645609415931859827734307762367947830110639831467043405387353622176899024" }},
   {{ "12.25", "-0.3273662654331959909797507119051924955282853934206185144087221653583658953364484322496982215172064596" }},
   {{ "12.28125", "-0.2931065748149348906573126400315634011327627839945547849105543140713203189089283254450491988430709458" }},
   {{ "12.3125", "-0.2594689972120933396662267471796226223140842016040887149108653318448105627021634377488137220685751418" }},
   {{ "12.34375", "-0.2263727011714726746425617255174464360665584672063083345910005855017649693977414015628137560181413471" }},
   {{ "12.375", "-0.1937415223592955920421472499371716132396126974936842056246045191659136204224066400221303821918510103" }},
   {{ "12.40625", "-0.1615032187034938758288345646836288312549849485871893783750506125037934982810946747034471565118071181" }},
   {{ "12.4375", "-0.1295887977200466002615144930375243133915724779506460096421464176334192279649808190617654665646655393" }},
   {{ "12.46875", "-0.09793190214949804406939767223801097423536917158984066549241688209747906679339832186909184479038423885" }},
   {{ "12.5", "-0.06646824186327419610199266949441387194390610792357612079517505380578030434009894147743014400122005453" }},
   {{ "12.53125", "-0.03513506141616282076636047393093268566965509833708523530838721391339373252314877573835767237228949322" }},
   {{ "12.5625", "-0.003870633688692445796420680716701608075985971581256241161345780831826430572573689071744247468488929075" }},
   {{ "12.59375", "0.02738622916941998236655243276631287278486472025157424329703232697741123275406245628528750156873296529" }},
   {{ "12.625", "0.05869665578128338112635128913606850911677921227526205464651894469274626817414600063862113806671147531" }},
   {{ "12.65625", "0.09012219459725165729474121609816933257302657281344601700769732387220992191083677289387048548334265912" }},
   {{ "12.6875", "0.1217252999603408610668897042413571531127645503736599822403036205555496676066474076056000689949341066" }},
   {{ "12.71875", "0.1535698333828775342772949638706552097777256705040011551078273258533143177119249497623834514512966756" }},
   {{ "12.75", "0.1857215886751401902594186441708939448272141506734588329272148156845153012668419405756499880982943312" }},
   {{ "12.78125", "0.2182488502657797346752690950871572837813792443000041521932413723748578742375061893600646885986149118" }},
   {{ "12.8125", "0.2512229950428828425999138094132092852018197118941615412720196242900796512859843472371170313836583775" }},
   {{ "12.84375", "0.2847191493692259054394416502657415422614131291273802036428533121457701035912610846112501083132620008" }},
   {{ "12.875", "0.3188169146481101964165914224483354040480603343474688683709071564169780696806924149291320418944830542" }},
   {{ "12.90625", "0.3536011770209831976646321735648611242564793780159081807119024532891168112397727872069370118132852613" }},
   {{ "12.9375", "0.389163019575800390066324389631248715515481435421443727622672043630746566664120384677948936473787333" }},
   {{ "12.96875", "0.4256007589819842824595519481972026675555296385735056063290277948532474177383987445148843331908874667" }},
   {{ "13", "0.4630211329364895955117902835081206683532120959519440470950206693928302579327807145125758116501924079" }},
   {{ "13.03125", "0.5015406704601738993466207755375547536965260852964643732753893793547109382261801451895601761930463122" }},
   {{ "13.0625", "0.5412872842615681585626378992160244527886510964571032692368478641312818490839188830849928560962861538" }},
   {{ "13.09375", "0.5824021335357886328662589223215248727421699777193264549116653046243519880397055116969487363618353957" }},
   {{ "13.125", "0.6250418172930577232327055575810817395750497037126950514585195397519699819130820731538599564259401139" }},
   {{ "13.15625", "0.6693809734298055686754246761212833674742679018564325405072602539903766997859347462285344333571366362" }},
   {{ "13.1875", "0.7156153783773706463324211230116191534127258818669295732823217991324981875804470761472072117982861964" }},
   {{ "13.21875", "0.7639656678174816067986696149135126827704120185065555912507382354963651332883430075609034401555571502" }},
   {{ "13.25", "0.8146818327614344736464451806034453710444224291621571573625236805118070729970340362557654528804594448" }},
   {{ "13.28125", "0.8680486902251570873258425734584921956065820604675624643643164481487468794096033758920520511975882962" }},
   {{ "13.3125", "0.9243925880092635749600582006440809362226783655074073519865821148035556209179629462013383469424274175" }},
   {{ "13.34375", "0.984089684751991735962930471741445607987192320278600727525448404432994693600540790050738561865588772" }},
   {{ "13.375", "1.04757625822842897028900134354672689006793126569323519874122370449523706721491998653522405351511" }},
   {{ "13.40625", "1.115361649708918497565229580465214098442269312226313131576538895853120727413355048723361997440716611" }},
   {{ "13.4375", "1.188044669282090096413968423531415025467187597047077575491037309801518019505454161493737933958668965" }},
   {{ "13.46875", "1.266334595496805761839284321117713547689424188079416196364317291844437171872585239983180116380091394" }},
   {{ "13.5", "1.351078347287010480371085577712699470101212829099407611657174914155576868564657173832374341089513584" }},
   {{ "13.53125", "1.443296057137394170028536177718664041051292527024487177527337760724543655637891131854085251072201818" }},
   {{ "13.5625", "1.544228244055125768572769325179666059641117940456933228593668266762756226316757718141030898582889678" }},
   {{ "13.59375", "1.65539925635608625536146268504299540984757874423396346584219270902415141849546774423536235044651061" }},
   {{ "13.625", "1.778703933748487611100295794480392286774372855795510955005500225382241572963111907877924106343034439" }},
   {{ "13.65625", "1.916528050760659646742064707886848214342582893475968412683323141359371632373241621750396872855231387" }},
   {{ "13.6875", "2.071918975654530318165994049833557729201414960634819277207514404216213840009455116234485076356302166" }},
   {{ "13.71875", "2.248832798150414958768559692131859458258869348641929756045263587943996112471489439485863476093034658" }},
   {{ "13.75", "2.452501131436276534259505261160229408535827308333544989748716942417182101185885609613251793631088723" }},
   {{ "13.78125", "2.689991141885544485574819071146851902487462547699495662222675009146133600868249015967786252574495347" }},
   {{ "13.8125", "2.971088997454883061146937345921981294715821069297105784485484470539822403438727250351950923586461683" }},
   {{ "13.84375", "3.309747852278236334086030022253642441335045150382837946015733966046635972988597769925369044448793434" }},
   {{ "13.875", "3.726571412380592611642200928513525816649689030739311675986112325390668863787780356510815096395312152" }},
   {{ "13.90625", "4.253314099623669334835259848593744934346239237141072606237802004691075608138259425654524049734148683" }},
   {{ "13.9375", "4.941607147834976432334070015221138910450692007921150173618524899066416996751652602822476062787651509" }},
   {{ "13.96875", "5.881399525150031955799636707531603412059246766703476508448423108137501182284498677930575806202198028" }},
   {{ "14", "7.244606616094805116868776904419943839744730261240134752545587919443539234559937291680749763590851956" }},
   {{ "14.03125", "9.406028188605279696133223875198323737712453281000019998207973537233195589286463672078393783342949057" }},
   {{ "14.0625", "13.36790967265482265511197119971831749461310263946778620293662813338523575832270133240885318460296722" }},
   {{ "14.09375", "23.0180098105772480411418077991835591801343058014965291845911306402355660736744079654039102037142972" }},
   {{ "14.125", "82.18587088008837479889498738727608856213126524631858084352266182200806013924238364295643998983071042" }},
   {{ "14.15625", "-52.39613928846073424250555787725052081495348867793023995281290229927733884543365803704289494278819236" }},
   {{ "14.1875", "-19.85087749437963381473636677142019044036915990708056158011806608744982156130940528545579140382954901" }},
   {{ "14.21875", "-12.23024032878110918179097238079573616977352169397336130087278904329232109578073694396971060087189843" }},
   {{ "14.25", "-8.825007825823643194479070661167279551172813025945214108644920017576189426949178857879125100439566237" }},
   {{ "14.28125", "-6.892346926954325909059194856822673062063097995636142714599547668355292064875194893124728097012275365" }},
   {{ "14.3125", "-5.644866498825540128381929944705578552248933555730964244899773528736917184084590171390597212873718595" }},
   {{ "14.34375", "-4.771610252795610046024685797665514766422880378783577882983742818079026166693261845053770481989424557" }},
   {{ "14.375", "-4.125051680440956814255997242162117938555651270642110578618995113301257483035180159821309310658332941" }},
   {{ "14.40625", "-3.626194314229776036712561256911661725676439848577743170654748551410882288678506147976699754338337632" }},
   {{ "14.4375", "-3.228918544653384166822369480885301355908586468628448310353628937744136710166892378083809771029038072" }},
   {{ "14.46875", "-2.904489208401961365438740405970863838028715766433251544998958231462070708355920879083964796103033733" }},
   {{ "14.5", "-2.634069132503832833409607035052858278935807513368801311481801280408668028920599740332367971681453643" }},
   {{ "14.53125", "-2.404794639925577553244012746285864594559572350297129379452932406519481483010765023196033751578709703" }},
   {{ "14.5625", "-2.207581089523679545128355853946391835592464237094323506839360879131859583353800754571132350944473674" }},
   {{ "14.59375", "-2.035829586659772321235017007716845648434117124540496962900856683971781311984497273615528750368081422" }},
   {{ "14.625", "-1.884630768865435904935595378906947231717012177189752377379403964925875302521199447518722408719775981" }},
   {{ "14.65625", "-1.750256190835124208623312849116061326812165846025415986783560739453729962659158203940996884358758139" }},
   {{ "14.6875", "-1.629822991609989147184189043054819627629510818289435539398133590083260226275976603564527843030512624" }},
   {{ "14.71875", "-1.521066667289361457373930052890176846963182405346389200320517277799503677666772192617687500257521097" }},
   {{ "14.75", "-1.42218336083774197380789085745137021789867848279597800468450871698744869740438289092755542692570482" }},
   {{ "14.78125", "-1.331718058572613129017722401996663023341875099157808098995252549045562547044521935057437210333349552" }},
   {{ "14.8125", "-1.248483823169844883793467308858957913772012915051151771810074084965259104516379708093866536649015936" }},
   {{ "14.84375", "-1.171502454280969620050113939757468091964527953537892858826299161314598145955301346129774398985079725" }},
   {{ "14.875", "-1.09996022360423379094065298284739111500559223495806048377446355428125902586706149601540328954397664" }},
   {{ "14.90625", "-1.033174396383465269461739738697259215672589493553897443441659340076220701053186730617856132398548093" }},
   {{ "14.9375", "-0.9705675907155131163149460322607739770705545946870992276893226926978460501228871025636381103213358614" }},
   {{ "14.96875", "-0.9116479124475490508064737825953723368462225788875404013990634500526165327350899245784037598875471869" }},
   {{ "15", "-0.8559934009085187640261927782380296049897864414742962979687446281783012594486541122545103129259226012" }},
   {{ "15.03125", "-0.8032397302332296612432159755337281904284459733163994877022298441144059785603187789136637043263789466" }},
   {{ "15.0625", "-0.7530703960754939484978446924032015414437125508200990314934869704250553438069278960228507610120908637" }},
   {{ "15.09375", "-0.7052088187318415299725928408540094093554358958059088325375638362095004704432693353838375709463129551" }},
   {{ "15.125", "-0.659411937622644225218300893404169827076548649885412227383887901950221923526355502978543776575055119" }},
   {{ "15.15625", "-0.6154649762727536414801474029122625793755198617269250343988631456453788448149731706979351529235579577" }},
   {{ "15.1875", "-0.5731771332183040948793178896118867037269598773610412277642977614409075169432893108977656087532808068" }},
   {{ "15.21875", "-0.5323780107028691176788965668440942781647156607765742951342412648358988559400799809724920990286528734" }},
   {{ "15.25", "-0.4929146351866012621666401573415074523464609163340854530825224584649245974424596583470196176643838565" }},
   {{ "15.28125", "-0.4546489554738299415350063572461193751006928326854122963028434583535719327916356408447934270192962049" }},
   {{ "15.3125", "-0.4174557284251085623680148206370333657407709127949181945517726979253617292255491330234726669774559159" }},
   {{ "15.34375", "-0.3812207207302419203446046603506711987787377766806738457878506795815713138223264722913354548269137032" }},
   {{ "15.375", "-0.3458391695028422582485004568448026536356203654713446806415710694635885346246660520826948870993463705" }},
   {{ "15.40625", "-0.311214455551825073642663079591570618837743590849903018274057606911872122304533000910107606645650336" }},
   {{ "15.4375", "-0.2772569518534830079599506190692912755868921271423525814120233867695893487373304945292237740353757468" }},
   {{ "15.46875", "-0.2438830165543173054192587099181568413614402066070223778896735213919894343566763962867244846621486138" }},
   {{ "15.5", "-0.2110141052012553133863173741408514435037061655011970687100846880180438182550528565062101594891245018" }},
   {{ "15.53125", "-0.1785759811388596991279739233477573236616687666102227187311862264495771634608666297275574790232136991" }},
   {{ "15.5625", "-0.1464980063716879847555275878278898667317766571375499828106145375100216949704230935552550931052323183" }},
   {{ "15.59375", "-0.1147124978452713735943481847220917786442201384229215777788278473922030662630912256721375261600875605" }},
   }};

   unsigned max_err = 0;
   for(unsigned k = 0; k < data.size(); k++)
   {
      T val = tan(T(data[k][0]));
      T e = relative_error(val, T(data[k][1]));
      unsigned err = e.template convert_to<unsigned>();
      if(err > max_err)
      {
         max_err = err;
      }
      val = tan(-T(data[k][0]));
      e = relative_error(val, T(-T(data[k][1])));
      err = e.template convert_to<unsigned>();
      if(err > max_err)
      {
         max_err = err;
      }
   }
   std::cout << "Max error was: " << max_err << std::endl;

   if (has_poor_large_value_support<T>::value)
   {
      T bug_value = 12 / std::numeric_limits<T>::epsilon();
      for (unsigned i = 0; i < 20; ++i, bug_value *= 1.1)
      {
         BOOST_TEST(tan(bug_value) == 0);
      }
   }

}

int main()
{
#ifdef TEST_BACKEND
   test<boost::multiprecision::number<boost::multiprecision::concepts::number_backend_float_architype> >();
#endif
#ifdef TEST_MPF_50
   test<boost::multiprecision::mpf_float_50>();
   test<boost::multiprecision::mpf_float_100>();
#endif
#ifdef TEST_MPFR_50
   test<boost::multiprecision::mpfr_float_50>();
   test<boost::multiprecision::mpfr_float_100>();
#endif
#ifdef TEST_MPFI_50
   test<boost::multiprecision::mpfi_float_50>();
   test<boost::multiprecision::mpfi_float_100>();
#endif
#ifdef TEST_CPP_DEC_FLOAT
   test<boost::multiprecision::cpp_dec_float_50>();
   test<boost::multiprecision::cpp_dec_float_100>();
#ifndef SLOW_COMPLER
   // Some "peculiar" digit counts which stress our code:
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<65> > >();
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<64> > >();
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<63> > >();
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<62> > >();
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<61, long long> > >();
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<60, long long> > >();
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<59, long long, std::allocator<char> > > >();
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<58, long long, std::allocator<char> > > >();
   // Check low multiprecision digit counts.
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<9> > >();
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<18> > >();
#endif
#endif
#ifdef TEST_FLOAT128
   test<boost::multiprecision::float128>();
#endif
#ifdef TEST_CPP_BIN_FLOAT
   test<boost::multiprecision::cpp_bin_float_50>();
   test<boost::multiprecision::number<boost::multiprecision::cpp_bin_float<35, boost::multiprecision::digit_base_10, std::allocator<char>, boost::long_long_type> > >();
#endif
   return boost::report_errors();
}



